SUBROUTINE adhesion(dz, zvalue,  &
                    alicount,loopnumz)
!
!  Purpose:
!
!    To apply alias tensile deformation along z axis to the current structure for calculating
!    the universal binding energy of 3D or 2D materials
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    alicount  : The alicount_th step for alias deformation
!    loopnumz  : The step number for z axis
!    dz        : The displacement along z axis
!    zvalue    : The alias deformation will be applied to the atoms, whose z coordinate > zvalue
!    adhvect   : The vectors defining the unit cell of the system after alias tensile deformation
!    adhpos    : Three coordinates for each atom after alias tensile deformation
!    adhfix    : Three additional logical flags determining whether to allow changes of the coordinates or not
!
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: alicount,loopnumz

DOUBLE PRECISION :: dz, zvalue
DOUBLE PRECISION :: adhvect(3,3), adhpos(ATOMMAX,3)

CHARACTER(len=1) :: adhfix(ATOMMAX,3)

! To read the POSCAR file

inpos_name='ALIPOS0'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! To carte
CALL cartepos(alat,     &
              privect,  &
              natom,    &
              ncoord,   &
              pripos)

! To calculate the zvalue      
IF ( ABS(zvalue+1.0000) .LT. ZERO_TOLERANCE ) THEN
        CALL zvaluer(inpos_name,&
                     zvalue,    &
                     alat,      &
                     privect,   &
                     natom,     &
                     ncoord,    &
                     pripos)
ENDIF

IF ( ABS(privect(1,3)).LT.ZERO_TOLERANCE .AND.  &
     ABS(privect(2,3)).LT.ZERO_TOLERANCE ) THEN

        DO i=1, natom
           IF ( pripos(i,3) .GT. zvalue ) THEN
              adhpos(i,1)=pripos(i,1)
              adhpos(i,2)=pripos(i,2)
              adhpos(i,3)=pripos(i,3)+dz
           ELSE
              adhpos(i,1)=pripos(i,1)
              adhpos(i,2)=pripos(i,2)
              adhpos(i,3)=pripos(i,3)
           ENDIF
        ENDDO
        
        ! Change the third cell vector c: dz 
        
        DO i=1,3
           DO j=1,3
              adhvect(i,j)=privect(i,j)
           ENDDO
        ENDDO
                
        adhvect(3,3)=adhvect(3,3)+dz/alat

ELSE        

        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Lattice vectors a and b in x-y plane are required!                       |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP

ENDIF

! To direct
CALL directpos(alat,     &
               adhvect,  &
               natom,    &
               ncoord,   &
               adhpos)

! Selective dynamic      
DO i=1,natom
   IF ( (ABS(pripos(i,3)-zvalue).LE.ADHESION_RLX_RANGE) .OR.  &
        (ABS(pripos(i,3)+privect(3,3)-zvalue).LE.ADHESION_RLX_RANGE) .OR.  &
        (ABS(pripos(i,3)-privect(3,3)-zvalue).LE.ADHESION_RLX_RANGE) ) THEN
           adhfix(i,1)='T'
           adhfix(i,2)='T'
           adhfix(i,3)='T'
   ELSE
           adhfix(i,1)='F'
           adhfix(i,2)='F'
           adhfix(i,3)='F'
   ENDIF
ENDDO



! To output the results

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*)
WRITE(19,'(A,I03.3,A1,I03.3)') '  ALIAS TENSILE DEFORMATION      ', alicount, '/', loopnumz
WRITE(19,*) ' ----------------------------------------------------------------------------- '
WRITE(19,*) '| Before alias tensile deformation, the POS file:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ',(ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ',(natomi(i), i=1,ntype)
DO i=1, natom
   WRITE(19,'(A3,3F20.10)') '| ',(pripos(i,j), j=1,3)
ENDDO

WRITE(19,*) '| '
WRITE(19,*) '| -----------------------------------------'
WRITE(19,'(A,2F8.4)') ' | dz and zvalue (in Ang):', dz, zvalue
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| '

WRITE(19,*) '| After alias tensile deformation, the POS file:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ',(adhvect(i,j), j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ',(ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ',(natomi(i), i=1,ntype)
DO i=1, natom
   WRITE(19,'(A3,3F20.10,3A3)') '| ',(adhpos(i,j), j=1,3), (adhfix(i,j), j=1,3)
ENDDO

CLOSE(19)

! To write the POSCAR file

ndynamic=01
outpos_name='ALIPOSI'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              adhvect,         &
              ele_symbol,      & 
              ntype,           &
              natom,           &
              natomi,          & 
              ndynamic,        &
              ncoord,          &
              adhpos,          &
              adhfix)
            
! End of writing the POSCAR file

END SUBROUTINE adhesion
