SUBROUTINE bond_len(inpos_name, bondcut)
!
!  Purpose:
!
!    To analyze the bond information of the structure file AFFPOS_[istr0].vasp with a given 
!    value of bond_cut
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    bondcut   : The maximum value for Bond Length 
!    bond_num_aver: The bond number for the total crystal structure
!    bond_lgth_aver: The bond length for the total crystal structure
!    dist      : The distance between per atom and the atoms in 3x3x3 supercell
!    dist_direct: The coordinate difference (Direct mode) between per atom and the atoms in 3x3x3 supercell
!    dist_carte: The coordinate difference (Cartesian mode) between per atom and the atoms in 3x3x3 supercell
!    supercell_indx : The index matrix for 3x3x3 supercell
!
!    atom_indx : The atomic index, for which the atom in crystal forms bond with per atom
!    atom_atom_indx: The atomic index in the supercell_indx matrix, for which the atom in 3x3x3 supercell forms bond with per atom
!    bond_num_atom: The bond number of per atom
!    bond_lgth_atom: The bond length for per atom
!
!
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk 
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

! ! Part one

DOUBLE PRECISION :: bondcut
DOUBLE PRECISION :: bond_num_aver, bond_lgth_aver
DOUBLE PRECISION :: dist(ATOMMAX,ATOMMAX,27)
DOUBLE PRECISION :: dist_direct(ATOMMAX,ATOMMAX,27,3)
DOUBLE PRECISION :: dist_carte(ATOMMAX,ATOMMAX,27,3)
DOUBLE PRECISION,DIMENSION(:,:),ALLOCATABLE :: supercell_indx

! ! Part two

INTEGER :: atom_indx(ATOMMAX,ATOMMAX), bond_num_atom(ATOMMAX)
INTEGER :: atom_atom_indx(ATOMMAX,ATOMMAX)
DOUBLE PRECISION :: bond_lgth_atom(ATOMMAX,ATOMMAX)

IF ( bondcut .GT. 10.0000 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Too large BOND_CUT, i.e. BOND_CUT > 10.0 Ang!                            |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP 
ENDIF

! To read the POSCAR file

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

ALLOCATE(supercell_indx(27,3))
supercell_indx(:,:)=RESHAPE((/-1,-1,-1,-1,-1,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,  &
                              -1,-1,-1,0,0,0,1,1,1,-1,-1,-1,0,0,0,1,1,1,-1,-1,-1,0,0,0,1,1,1,  &
                              -1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1/),(/27,3/))

! To get the avarage of Bond Length for all atoms

bond_lgth_aver=0.00D0
bond_num_aver=0.00D0

DO i=1,natom
   DO j=1,natom
      DO kk=1,27
         DO k=1,3
            dist_direct(i,j,kk,k)=pripos(i,k)-pripos(j,k)+supercell_indx(kk,k)
         ENDDO

         DO k=1,3
            dist_carte(i,j,kk,k)=alat*(dist_direct(i,j,kk,1)*privect(1,k)  &
                                      +dist_direct(i,j,kk,2)*privect(2,k)  &
                                      +dist_direct(i,j,kk,3)*privect(3,k))
         ENDDO

         dist(i,j,kk)=SQRT(dist_carte(i,j,kk,1)**2+dist_carte(i,j,kk,2)**2+dist_carte(i,j,kk,3)**2)

         IF ( dist(i,j,kk).GT.0 .AND. dist(i,j,kk).LT.bondcut ) THEN
                 bond_lgth_aver=bond_lgth_aver+dist(i,j,kk)
                 bond_num_aver=bond_num_aver+1.00D0
         ENDIF
      ENDDO
   ENDDO
ENDDO

DEALLOCATE(supercell_indx)

IF ( ABS(bond_num_aver) .GT. ZERO_TOLERANCE ) THEN
        bond_lgth_aver=bond_lgth_aver/bond_num_aver
        bond_num_aver=bond_num_aver/natom
ELSE
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Too small BOND_CUT, i.e., no bond between atoms!                         |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP 
ENDIF

! To get the Bond Length for per atom

DO i=1,natom

   bond_num_atom(i)=0.00D0

   DO j=1,natom
      DO kk=1,27
         IF ( dist(i,j,kk).GT.0.00D0 .AND. dist(i,j,kk).LT.bondcut ) THEN
                 bond_num_atom(i)=bond_num_atom(i)+1
                 atom_indx(i,bond_num_atom(i))=j
                 bond_lgth_atom(i,bond_num_atom(i))=dist(i,j,kk)
                 atom_atom_indx(i,INT(bond_num_atom(i)))=kk
         ENDIF
      ENDDO
   ENDDO
ENDDO

! To write the results

! WRITE(*,'(A35,2F8.4)') 'The Average Bond Number Per Atom and Length:', bond_num_aver, bond_lgth_aver
! WRITE(*,*)
!
! DO i=1,natom
!    WRITE(*,'(A5,I03.3,A1,I4,$)') 'ATOM(', i, ')', bond_num_atom(i)
!    DO j=1,bond_num_atom(i)
!       WRITE(*,'(A3,I03.3,A1,F8.4,$)') '  (', atom_indx(i,j), ')', bond_lgth_atom(i,j)
!    ENDDO
!    WRITE(*,*)
! ENDDO

! To write the file BONDCAR

OPEN(UNIT=67, FILE='BONDCAR', STATUS='unknown')

! WRITE(67,*) ten_str, shear_str, istr0, step_length, step_num
DO i=1,natom
   WRITE(67,'(A5,I03.3,A1,I4,$)') 'ATOM(', i, ')', bond_num_atom(i)
   DO j=1,bond_num_atom(i)
      WRITE(67,'(A3,I03.3,A1,I4,$)') '  (', atom_indx(i,j), ')', atom_atom_indx(i,j)
   ENDDO
   WRITE(67,*)
ENDDO

CLOSE(67)

END SUBROUTINE bond_len
