SUBROUTINE indxproj(pvh, pvk, pvl,  &
                    uvu, uvv, uvw)
!
!  Purpose:
!
!    Crystallographic index method of projection
!
!    To project one specific orientation (i.e., the imposed strain direction) of a given crystal
!    to be parallel to one axis vector (e.g. y) for affine tensile deformation, and projects 
!    one axis vector (e.g. x) to be perpendicular to the slip plane and another one (e.g. y) to
!    be parallel to the slip direction in that plane for the affine shearing deformation
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    pvh, pvk, pvl: Projection vector [pvh pvk pvl] along X axis
!    uvu, uvv, uvw: Upward vector [uvu uvv uvw] along Y axis
!    oldX(Y,Z) :
!    newX(Y,Z) :
!    recipvect : The reciprocal vector of privect
!    projmat   : The projection matrix, i.e., privect=indxvect*projmat
!    projmatI  : The inverse matrix of projmat, i.e., indxvect=privect*projmatI
!    indxvect  : The vectors defining the unit cell of the system after projection
!
!
!  Note:
!    Two crystallographic directions
!        
!       The former: projection vector along X axis that is normal to (pvh pvk pvl) plane
!       The latter: upward vector along Y axis, i.e. [uvu uvv uvw] 
!
!       e.g. projection vector along X axis is x[pvh pvk pvl]
!            upward vector along Y axis is     y[uvu uvv uvw]
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: pvh, pvk, pvl
INTEGER :: uvu, uvv, uvw

DOUBLE PRECISION :: oldX(3), oldY(3), oldZ(3)
DOUBLE PRECISION :: newX(5), newY(5), newZ(5)
DOUBLE PRECISION :: recipvect(3,3), tmpvect(5,5)
DOUBLE PRECISION :: projmat(3,3), projmatI(3,3), indxvect(3,3)

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

! Define the oldY and oldX

oldX(1)=pvh
oldX(2)=pvk
oldX(3)=pvl
oldY(1)=uvu
oldY(2)=uvv
oldY(3)=uvw
oldZ(1)=0.0000
oldZ(2)=0.0000
oldZ(3)=0.0000

IF ( oldX(1)*oldY(1)+oldX(2)*oldY(2)+oldX(3)*oldY(3).NE.0 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE') 
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Upward vector is not in projected plane!                                 |'
        WRITE(19,*) '|    When the lattice vector [uvw] lies on the (hkl) plane, u, v, w, h, k     |'
        WRITE(19,*) '|    and l must satisfy the condition:                                        |'
        WRITE(19,*) '|                                   hu+kv+lw=0                                |'
        WRITE(19,*) '|    This condition must, therefore, be satisfied in order to specify the     |'
        WRITE(19,*) '|    upward direction on the screen, otherwise the upward direction on the    |'
        WRITE(19,*) '|    screen is automatically determined by this code.                         |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP
ENDIF

! define the reciprocal vector of privect

DO i=1,3
   DO j=1,3
      tmpvect(i,j)=privect(i,j)
   ENDDO
ENDDO
DO i=1,3
   DO j=4,5
      tmpvect(i,j)=tmpvect(i,j-3)
   ENDDO
ENDDO
DO i=4,5
   DO j=1,5
      tmpvect(i,j)=tmpvect(i-3,j)
   ENDDO
ENDDO

DO i=1,3
   DO j=1,3
      recipvect(i,j)=tmpvect(i+1,j+1)*tmpvect(i+2,j+2)  &
                    -tmpvect(i+1,j+2)*tmpvect(i+2,j+1)
   ENDDO
ENDDO

! define the newY and newX relative of XYZ

DO i=1,3
   newX(i)=0.0000
   DO j=1,3
      newX(i)=newX(i)+oldX(j)*recipvect(j,i)
   ENDDO
ENDDO

DO i=1,3
   newY(i)=0.0000
   DO j=1,3
      newY(i)=newY(i)+oldY(j)*privect(j,i)
   ENDDO
ENDDO

! define the newZ relative of XYZ 

DO i=4,5
   newX(i)=newX(i-3)
   newY(i)=newY(i-3)
ENDDO

DO i=1,3
   newZ(i)=newX(i+1)*newY(i+2)-newY(i+1)*newX(i+2)
ENDDO

! define the transformation matrix

DO i=1,3
   projmat(1,i)=newX(i)/SQRT(newX(1)**2+newX(2)**2+newX(3)**2)
   projmat(2,i)=newY(i)/SQRT(newY(1)**2+newY(2)**2+newY(3)**2)
   projmat(3,i)=newZ(i)/SQRT(newZ(1)**2+newZ(2)**2+newZ(3)**2)
ENDDO

! WRITE(*,*) 'Orientation matrix is as follows:'
! DO i=1,3
!    WRITE(*,'(3F20.10)') (projmat(i,j),j=1,3)
! ENDDO

! Transport the orientation matrix

! >>> projmatI=projmat^-1
!
! "projmat" is a unitary matrix 
! "projmatI" is equal to the transport matrix of "projmat" 

DO i=1,3
   DO j=1,3
      projmatI(i,j)=projmat(j,i)
   ENDDO
ENDDO

! To get new vect after the transform the direction

! To operate the lattice vectors by matrix
! ---------------------------------------------
!       |a11 a12 a13|       |mat11 mat12 mat13|
!  R=   |a21 a22 a23|  MAT= |mat21 mat22 mat23|
!       |a31 a32 a33|       |mat31 mat32 mat33|
! ---------------------------------------------
!       |aa11 aa12 aa13|
!  RR=  |aa21 aa22 aa23|
!       |aa31 aa32 aa33|
! ---------------------------------------------
!  RR= R*MAT
!  Notice: aa11=a11*mat11+a12*mat21+a13*mat31
!          aa12=a11*mat12+a12*mat22+a13*mat32
!          aa13=a11*mat13+a12*mat23+a13*mat33
!  Notice: aa21=a21*mat11+a22*mat21+a23*mat31
!          aa22=a21*mat12+a22*mat22+a23*mat32
!          aa23=a21*mat13+a22*mat23+a23*mat33
!  Notice: aa31=a31*mat11+a32*mat21+a33*mat31
!          aa32=a31*mat12+a32*mat22+a33*mat32
!          aa33=a31*mat13+a32*mat23+a33*mat33
! ---------------------------------------------
!  In VESTA, orientation is the inverse of MAT
!  i.e. the 1st row corresponds to mat11 mat21 mat31
!       the 1st col corresponds to mat11 mat12 mat13 
! ---------------------------------------------
! 
!  For "projmat":  privect=indxvect*projmat
!  For "projmatI": indxvect=privect*projmatI 
!  

DO i=1,3
   DO j=1,3
      indxvect(i,j)=0.0000
      DO k=1,3
         indxvect(i,j)=indxvect(i,j)+privect(i,k)*projmatI(k,j)
      ENDDO
   ENDDO
ENDDO  

! To output the results

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
WRITE(19,*)
WRITE(19,*) ' CRYSTAL PROJECTION (INDXPROJ)'
WRITE(19,*) ' ----------------------------------------------------------------------------- '
WRITE(19,*) '| Before Projction, the three lattice vectors:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO

WRITE(19,*) '| '
WRITE(19,*) '| -----------------------------------------'
WRITE(19,'(A,3I3,A1)') ' | projection vector along X axis is x[',pvh, pvk, pvl,']'
WRITE(19,'(A,3I3,A1)') ' | upward vector along Y axis is     y[',uvu, uvv, uvw,']'
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| Orientation matrix is as follows:'
DO i=1,3
   WRITE(19,"(A11,3F12.6)") '|         ', (projmat(i,j),j=1,3)
ENDDO
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| '

WRITE(19,*) '| After Projction, the three lattice vectors:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (indxvect(i,j), j=1,3)
ENDDO

CLOSE(19)

! To write the POSCAR file

outpos_name='PROJPOS'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              indxvect,        &
              ele_symbol,      & 
              ntype,           &
              natom,           &
              natomi,          & 
              ndynamic,        &
              ncoord,          &
              pripos,          &
              prifix)
            
! End of writing the POSCAR file

END SUBROUTINE indxproj
