SUBROUTINE kpta(kppra,  &
                kscheme)
!
!  Purpose:
!
!    To create the KPOINTS file according the "K-Point Per Reciprocal Atom"
!    (KPPRA). This is a way to automatically set the k-point mesh for a number
!    of similar systems. The mesh along the three axes is automatically chosen 
!    to make the mesh as uniform as possible.
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    kppra     : The input kppra value
!    kshame    : the type mesh to use: Gamma-centered or Monkorst-Pack. ("m" "M" or "g" "G")
!    imesh     : The calculated k_i values based on the kspac value
!    recipvect : The reciprocal lattice vectors of privect in VASP
!    crossvect : The cross product of recipvect(i,j)
!    dotnorm   : dotnorm(i)=ABS(recipvect(i,1)*crossvect(i,1)+recipvect(i,2)*crossvect(i,2)+recipvect(i,3)*crossvect(i,3))
!    crossnorm : crossnorm(i)=SQRT(crossvect(i,1)**2+crossvect(i,2)**2+crossvect(i,3)**2)
!    kpt_indx  : kpt_indx(i)=dotnorm(i)/crossnorm(i)
!
!
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: kppra
INTEGER :: imesh(3)

DOUBLE PRECISION :: nb_kpts, bestrf, normalizer
DOUBLE PRECISION :: kpt_indx(3), fmesh(3)
DOUBLE PRECISION :: dotnorm(3), crossnorm(3)
DOUBLE PRECISION :: recipvect(5,5), tmpvect(3,3), crossvect(3,3)

CHARACTER(len=1) :: kscheme

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

OPEN(UNIT = 19, POSITION = 'Append', FILE='OUTFILE') 

WRITE(19,*)
WRITE(19,*) ' KPPRA METHOD'
WRITE(19,*) ' ----------------------------------------------------------------------------- '

WRITE(19,*) '| After reading, the three lattice vectors:'
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.15)') '| ', (privect(i,j),j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ', (ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ', (natomi(i), i=1,ntype)

WRITE(19,*) '| -----------------------------------------'
WRITE(19,'(A,I5)') ' | K-Point Per Reciprocal Atom is ', kppra
WRITE(19,*) '| -----------------------------------------'

CLOSE(19)

! define the reciprocal vector of privect

DO i=1,3
   DO j=1,3
      tmpvect(i,j)=privect(i,j)
   ENDDO
ENDDO

CALL brinv(tmpvect,3)

DO i=1,3
   DO j=1,3
      ! Note that a transposition is applied to tmpvect
      recipvect(i,j)=2*PI*tmpvect(j,i)
   ENDDO
ENDDO

DO i=4,5
   DO j=1,3
      recipvect(i,j)=recipvect(i-3,j)
   ENDDO
ENDDO

DO i=1,5
   DO j=4,5
      recipvect(i,j)=recipvect(i,j-3)
   ENDDO
ENDDO

! determine the kpt_indx(i)

DO i=1,3
   DO j=1,3
      crossvect(i,j)=recipvect(i+1,j+1)*recipvect(i+2,j+2)-recipvect(i+1,j+2)*recipvect(i+2,j+1)
   ENDDO
ENDDO

DO i=1,3
   dotnorm(i)=0.0000
   DO j=1,3
      dotnorm(i)=dotnorm(i)+recipvect(i,j)*crossvect(i,j)
   ENDDO
   dotnorm(i)=ABS(dotnorm(i))
ENDDO

DO i=1,3
   crossnorm(i)=0.0000
   DO j=1,3
      crossnorm(i)=crossnorm(i)+crossvect(i,j)*crossvect(i,j)
   ENDDO
   crossnorm(i)=SQRT(crossnorm(i))
ENDDO

DO i=1,3
   kpt_indx(i)=dotnorm(i)/crossnorm(i)
ENDDO

nb_kpts=kppra/natom
normalizer=(kpt_indx(1)*kpt_indx(2)*kpt_indx(3)/nb_kpts)**(1.0000/3.0000)
DO i=1,3
   imesh(i)=NINT(kpt_indx(i)/normalizer)
   fmesh(i)=kpt_indx(i)/normalizer-REAL(imesh(i))
ENDDO

! Floor and Ceiling

! IF ( kscheme.NE.'m' .AND. kscheme.NE. 'g' .AND.  &
!      kscheme.NE.'M' .AND. kscheme.NE. 'G' ) THEN
! 
!         OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
!         WRITE(19,*) 'Error!!! NOT "M" OR "G" FOR KSCHEME'
!         CLOSE(19)
!         STOP
! ENDIF

IF( kscheme.EQ.'m' .OR. kscheme.EQ.'M' ) THEN 
        DO i=1,3
           IF( MOD(imesh(i),2) .EQ. 1 ) THEN
                   imesh(i)=imesh(i)+1
                   fmesh(i)=0.0000
           ENDIF
        ENDDO
ENDIF

DO 200, WHILE ((imesh(1)*imesh(2)*imesh(3)) .LT. nb_kpts)
        
bestrf=0.0000

DO i=1,3
   IF (fmesh(i) .GT. bestrf) THEN
           bestrf=fmesh(i)
   ENDIF
ENDDO

DO i=1,3
   IF ( ABS(fmesh(i)-bestrf) .LT. ZERO_TOLERANCE ) THEN
           imesh(i)=imesh(i)+1
           IF( kscheme.EQ.'m' .OR. kscheme.EQ.'M' ) THEN
                imesh(i)=imesh(i)+1
           ENDIF
           fmesh(i)=0.0000
   ENDIF
ENDDO
200     CONTINUE

! To avoid imesh(i)==0

DO i=1,3
   IF( imesh(i) .EQ. 0 ) THEN
           imesh(i)=imesh(i)+1
   ENDIF
ENDDO

! Writing the kemsh(i)

OPEN(UNIT = 19,POSITION = 'Append',FILE = 'OUTFILE') 

WRITE(19,*) '| '
WRITE(19,*) '| Automatic mesh'
WRITE(19,*) '|   0'
IF(kscheme.EQ.'m' .OR. kscheme.EQ.'M') THEN
        WRITE(19,*)  '| Monkhorst-pack'
ELSE
        write(19,*)  '| Gamma-centered'
ENDIF
WRITE(19,'(A3,3I3)') '| ', (imesh(i), i=1,3)
WRITE(19,*) '|   0  0  0'

CLOSE(19)

! To write the NEWKPT file

OPEN(UNIT=30, FILE='NEWKPT', STATUS = 'unknown')

WRITE(30,'(A14)') 'Automatic mesh'
WRITE(30,'(A3)') '  0'
IF(kscheme.EQ.'m' .OR. kscheme.EQ.'M') THEN
        WRITE(30,'(A14)') 'Monkhorst-pack'
ELSE
        WRITE(30,'(A14)') 'Gamma-centered'
ENDIF
WRITE(30,'(3I3)') (imesh(i), i=1,3)
WRITE(30,'(A9)') '  0  0  0'

CLOSE(30)

END SUBROUTINE kpta
