!------------------------------------------------------------------
!     Copyright[c] 2018-2019, S. H. Zhang and R. F. Zhang        
!     ADAIS - an efficient open source command-line program 
!             for Automatic Derivation of Anisotropic Ideal 
!             Strength via high-throughput first principles 
!             computations. It is designed by Dr. S.H. Zhang 
!             and Prof. R.F. Zhang at Beihang University.                  
!                                                                
!     This program is free software: you can redistribute it     
!     and/or modify it under the terms of the BSD 3-Clause       
!     License as published by the Free Software Foundation,      
!     either version 3 of the License, or (at your option)       
!     any later version.                                        
!------------------------------------------------------------------

MODULE constants
!
!  Variables:
!
!    ATOMMAX: Maximal total number of atoms
!    TYPEMAX: Maximal number of atomic type
!    STRAINMAX: Maxinal number of affine strain step
!    PI: The valu of PI
!    ZERO_TOLERANCE: Two floating numbers A and B are considered 
!                    equal, as |A-B|<ZERO_TOLERANCE
!    SYMPREC: Tolerance of distance between atomic positions and 
!             the lengths of lattice vectors to be tolerated in 
!             the symmetry finding
!    ADHESION_RLX_RANGE: The selective dynamics of the atoms, for 
!                        which the distance away from the cleavage 
!                        plane is ADHESION_RLX_RANGE, is set as 
!                        "T T T", for the others, it is set as 
!                        "F F F"
!    BOND_CUT: The maximum value for Bond Length
!

IMPLICIT NONE

INTEGER, PARAMETER :: ATOMMAX=1000
INTEGER, PARAMETER :: TYPEMAX=10
INTEGER, PARAMETER :: STRAINMAX=100
DOUBLE PRECISION, PARAMETER :: PI=3.1415926
DOUBLE PRECISION, PARAMETER :: ZERO_TOLERANCE=1E-4
DOUBLE PRECISION, PARAMETER :: SYMPREC=0.0100
DOUBLE PRECISION, PARAMETER :: ADHESION_RLX_RANGE=4.0000
! DOUBLE PRECISION, PARAMETER :: BOND_CUT=0.0000

END MODULE constants

PROGRAM main
!
!  Purpose:
!
!    The mian function of ADAIS
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    inpos_mane: Name of input file
!    outpos_name: Name of output file
!
!    pos_title : 'name' of the system
!    alat      : A universal scaling factor, which is used to scale all lattice vectors and all atomaic coordinates
!    privect   : The vectors defining the unit cell of the system
!    ele_symbol: 
!    ntype     : Number of atomic type
!    natom     : Total number of atoms
!    natomi    : Number of atoms per atomic species
!    ndynamic  : Ndynamic=1: Selective Dynamics switched on; ndynamic=0: Selective Dynamics switched off
!    ncoord    : Ncoord=1: Cartesian mode; ncoord=0: Direte mode
!    pripos    : Three coordinates for each atom
!    prifix    : Three additional logical flags determining whether to allow changes of the coordinates or not
!    atom_type : Array for atomic type using in SPGLIB
!
!    pricoord  : 'C' or 'c': Cartesian mode; 'D' or 'd': Direte mode
!    select_dynamic: 'S' or 's': Selective Dynamics switched on
!
!    --------------------------------------------------------------------------------------------------
!
!    option: variable for input getting command argument (GET_COMMAND_ARGUMENT)
!
!    affcount  : The affcount_th step for affine deformation
!    affopt    : Five input arguments for affine deformation, i.e., ten_str, shear_str, istr0, step_length and step_num
!    disturbing: Whether apply disturbance as apply affine stran (disturbing=1) or NOT (disturbing=0)
!    distopt   : Four input arguments for applying disturbance, i.e., distance, dxx, dyy and dzz
!    strain    : The values of tensile and shearing strain
!    istrvalue : The accumulated strain value
!    istr0name : The name of initial POS file for affine deformation, i.e., AFFPOS_istr0.vasp
!    ten_str   : The weight for tensile strain, i.e., strain(1)=step_length*ten_str
!    shear_str : The weight for shearing strain, i.e., strain(2)=step_length*shear_str
!    istr0     : The initial strain value
!    step_length: The step length for affine deformation
!    step_num  : The step number for affine deformation
!
!    alicount  : The alicount_th step for alias deformation
!    loopnumx(y,z): The step number for x (y, z) axis
!    istartx(y,z): The initial alias strain value
!    iendx(y,z): The end alias strain value
!    ispacingx(y,z): The spacing value for alias strain
!    dx(y,z)   : The displacement along x (y,z) axis
!    zvalue    : The alias deformation will be applied to the atoms, whose z coordinate > zvalue
!
!    matopt    : The projection matrix (3x3)
!    rotopt    : The rotation angle of x, y and z axes
!    indxopt   : projection vector [pvh pvk pvl] along X axis and upward vector [uvu uvv uvw] along Y axis
!    newlat    : The new lattice vector for redefining lattive vector (3x3) 
!
!    kmesh     : Three k_i values for creating kpt file via kmesh method
!    kppra     : The kppra value for creating kpt file via kppra method
!    kspac     : The kspacing value for creating kpt file via kspac method
!    kscheme   : The type mesh to use: Gamma shift or Monkorst-Pack. ("mp" or "g")
!    pressure  : The pressure value for vreacing incar file, i.e. PSTRESS=pressure*10
!    sympreci  : The input tolerance of distance between atomic positions and the lengths of lattice vectors to be tolerated in the symmetry finding
!    twintype  : The twin type for creating twin structure
!    sup_range : The range of supercell
!    layer_spac: The layer spacing for creating twin structure via twintype=1
!    bondcut   : The input maximum value for Bond Length
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=1) :: pricoord, select_dynamic
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

LOGICAL :: filefound

! The variables for main program

CHARACTER(len=20) :: option(20)

        !! For affine deformation
        INTEGER :: countnum
        INTEGER :: affcount, disturbing
        DOUBLE PRECISION :: affopt(5), distopt(4)
        DOUBLE PRECISION :: ten_str, shear_str, istr0, step_length, step_num
        DOUBLE PRECISION :: strain(2), istrvalue
        CHARACTER(len=30) :: istr0name
        CHARACTER(len=30) :: istr0name_tmp, inpos_name_tmp, outpos_name_tmp
        CHARACTER(len=20) :: titstr1, titstr2
        CHARACTER(len=20) :: tmpstr
        CHARACTER(len=100) :: str_tmp        

        !! For alias deformation
        INTEGER :: alicount
        INTEGER :: loopnumx, loopnumy, loopnumz
        DOUBLE PRECISION :: aliopt(6)
        DOUBLE PRECISION :: istartx, iendx, ispacingx
        DOUBLE PRECISION :: istarty, iendy, ispacingy
        DOUBLE PRECISION :: istartz, iendz, ispacingz
        DOUBLE PRECISION :: zvalue, zvalue0
        DOUBLE PRECISION :: dxr, dyr, dzr
        DOUBLE PRECISION :: dx, dy, dz

        !! For proj and newlat
        INTEGER :: indxopt(6) !, newlat(3,3)
        DOUBLE PRECISION :: matopt(3,3), rotopt(3)
        DOUBLE PRECISION :: newlat(3,3)

        !! For others        
        INTEGER :: kmesh(3), kppra
        INTEGER :: twintype
        INTEGER :: sup_range(3)
        DOUBLE PRECISION :: kspac
        DOUBLE PRECISION :: pressure
        DOUBLE PRECISION :: sympreci
        DOUBLE PRECISION :: bondcut
        DOUBLE PRECISION :: layer_spac
        CHARACTER(len=20) :: kscheme

CALL GET_COMMAND_ARGUMENT(1,option(1))
! IF ( LEN_TRIM(option(1) .EQ. 0 ) THEN
!         EXIT
! ENDIF

! ADAIS --affine -pure3d/-simp3d/-unia2d/-biax2d ten_str shear_str istr0 step_length step_num [-disturbing distance dxx dyy dzz]
IF ( option(1) .EQ. '--affine' ) THEN

        CALL GET_COMMAND_ARGUMENT(2,option(2))
        DO i=1,5
           CALL GET_COMMAND_ARGUMENT(i+2,option(i+2))
           READ(option(i+2),*) affopt(i)
        ENDDO

        INQUIRE( FILE='LOOPCAR', EXIST=filefound )
        ! Strat from istri
        IF ( filefound ) THEN
                OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
                READ(87,*) titstr1, titstr2

                IF ( TRIM(titstr1).EQ.'AFFINE' .AND. TRIM(titstr2).EQ.TRIM(option(2)(2:7)) ) THEN
                        READ(87,*)
                        READ(87,*) tmpstr, tmpstr, tmpstr, ten_str
                        READ(87,*) tmpstr, tmpstr, tmpstr, shear_str
                        READ(87,*) tmpstr, tmpstr, tmpstr, istr0
                        READ(87,*) tmpstr, tmpstr, tmpstr, step_length
                        READ(87,*) tmpstr, tmpstr, tmpstr, step_num

                        IF( ABS(ten_str    -affopt(1)) .LT. ZERO_TOLERANCE .AND. &
                            ABS(shear_str  -affopt(2)) .LT. ZERO_TOLERANCE .AND. &
                            ABS(istr0      -affopt(3)) .LT. ZERO_TOLERANCE .AND. &
                            ABS(step_length-affopt(4)) .LT. ZERO_TOLERANCE .AND. &
                            ABS(step_num   -affopt(5)) .LT. ZERO_TOLERANCE ) THEN
        
                                READ(87,*) tmpstr, tmpstr, tmpstr, affcount
                                affcount=affcount+1
                                
                                IF ( affcount .EQ. 1 ) THEN
                                        ! To read the POSCAR file
                                        inpos_name='CONTCAR'
                                        CALL readpos(inpos_name,      &
                                                     pos_title,       &
                                                     alat,            &
                                                     privect,         &
                                                     ele_symbol,      & 
                                                     ntype,           &
                                                     natom,           &
                                                     natomi,          & 
                                                     ndynamic,        &
                                                     ncoord,          &
                                                     pripos,          &
                                                     prifix,          &
                                                     atom_type)
                        
                                        ! To write the POSCAR file
                                        WRITE(outpos_name,'(A7,F6.4,A5)') 'AFFPOS_', 0.0000, '.vasp'
                                        CALL writepos(outpos_name,     &
                                                      pos_title,       &
                                                      alat,            &
                                                      privect,         &
                                                      ele_symbol,      & 
                                                      ntype,           &
                                                      natom,           &
                                                      natomi,          & 
                                                      ndynamic,        &
                                                      ncoord,          &
                                                      pripos,          &
                                                      prifix)
                                ENDIF
        
                        ELSE
                                affcount=1
                        ENDIF
                ELSE
                        affcount=1
                ENDIF
                CLOSE(87)
        ! Start from istr0
        ELSE
                affcount=1
        ENDIF

        IF( affcount .EQ. 1 ) THEN
                ! Whether the ISTR0POS exists or Not

                IF ( ABS(affopt(3)) .GT. ZERO_TOLERANCE ) THEN
                        WRITE(istr0name_tmp,'(A7,F8.4,A5)') 'AFFPOS_', affopt(3), '.vasp'
                        
                        countnum=0
                        istr0name='                              '
                        DO kk=1,LEN_TRIM(istr0name_tmp)
                           IF ( istr0name_tmp(kk:kk) .NE. ' ' ) THEN
                                   countnum=countnum+1
                                   istr0name(countnum:countnum)=istr0name_tmp(kk:kk)
                           ENDIF
                        ENDDO
                        istr0name=TRIM(istr0name)

                        INQUIRE( FILE=istr0name, EXIST=filefound )
                        IF ( .NOT.filefound ) THEN
                                str_tmp=TRIM(istr0name)//' does not exit for affine deformation!'
                                DO i=(LEN_TRIM(str_tmp)+1),100
                                   str_tmp(i:i)=' '
                                ENDDO

                                OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
                                WRITE(19,*)
                                WRITE(19,*) ' ----------------------------------------------------------------------------- '
                                WRITE(19,*) '|                                     ERROR                                   |'
                                WRITE(19,*) '|                            -----------------------                          |'
                                WRITE(19,'(A25, A, A5)') '|    The istr0file file ', str_tmp(1:50), '    |'
                                IF ( LEN_TRIM(str_tmp) .GT. 50 ) THEN
                                        WRITE(19,'(A6, A, A34)') '|    ', str_tmp(51:100), '    |'
                                ENDIF
                                WRITE(19,*) ' ----------------------------------------------------------------------------- '
                                CLOSE(19)
                                STOP
                        ENDIF
                ELSE
                        WRITE(istr0name,'(A7,F6.4,A5)') 'AFFPOS_', 0.0000, '.vasp'
                        INQUIRE( FILE=istr0name, EXIST=filefound )
                        IF ( .NOT.filefound ) THEN
                                istr0name='AFFPOS0'
                                INQUIRE( FILE=istr0name, EXIST=filefound )
                                IF ( filefound ) THEN
                                        affcount=0
                                ELSE
                                        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
                                        WRITE(19,*)
                                        WRITE(19,*) ' ----------------------------------------------------------------------------- '
                                        WRITE(19,*) '|                                     ERROR                                   |'
                                        WRITE(19,*) '|                            -----------------------                          |'
                                        WRITE(19,*) '|    The istr0file file of AFFPOS_0.0000.vasp or AFFPOS0 does not exit as     |'
                                        WRITE(19,*) '|    istr0=0.0000 for affine deformation!                                     |'
                                        WRITE(19,*) ' ----------------------------------------------------------------------------- '
                                        CLOSE(19)
                                        STOP
                                ENDIF
                        ENDIF                
                ENDIF

        ENDIF        

        ! To define the tensile and shearing strain value

        IF ( affcount .EQ. 0 ) THEN

                DO i=1,2
                   strain(i)=0.0000
                ENDDO

        ELSEIF (affcount .EQ. 1) THEN

                DO i=1,2
                   strain(i)=affopt(i)*((1.0000+affopt(3)+affopt(4)*affcount)  &
                                       /(1.0000+affopt(3)+affopt(4)*(affcount-1))-1.0000)
                ENDDO
               
                ! To read the POSCAR file 
                WRITE(inpos_name_tmp,'(A7,F8.4,A5)') 'AFFPOS_', affopt(3), '.vasp'

                countnum=0
                inpos_name='                              '
                DO kk=1,LEN_TRIM(inpos_name_tmp)
                   IF ( inpos_name_tmp(kk:kk) .NE. ' ' ) THEN
                           countnum=countnum+1
                           inpos_name(countnum:countnum)=inpos_name_tmp(kk:kk)
                   ENDIF
                ENDDO
                inpos_name=TRIM(inpos_name)

                CALL readpos(inpos_name,      &
                             pos_title,       &
                             alat,            &
                             privect,         &
                             ele_symbol,      & 
                             ntype,           &
                             natom,           &
                             natomi,          & 
                             ndynamic,        &
                             ncoord,          &
                             pripos,          &
                             prifix,          &
                             atom_type)                

                ! To write the POSCAR file
                outpos_name='AFFPOS0'
                CALL writepos(outpos_name,     &
                              pos_title,       &
                              alat,            &
                              privect,         &
                              ele_symbol,      & 
                              ntype,           &
                              natom,           &
                              natomi,          & 
                              ndynamic,        &
                              ncoord,          &
                              pripos,          &
                              prifix)                

        ELSEIF ( affcount .GT. 1 ) THEN

                DO i=1,2
                   strain(i)=affopt(i)*((1.0000+affopt(3)+affopt(4)*affcount)  &
                                       /(1.0000+affopt(3)+affopt(4)*(affcount-1))-1.0000)
                ENDDO
        
                ! Whether CONTCAR file exists or Not
                INQUIRE( FILE='CONTCAR', EXIST=filefound )
                IF ( filefound ) THEN
        
                        ! To read the POSCAR file
                        inpos_name='CONTCAR'
                        CALL readpos(inpos_name,      &
                                     pos_title,       &
                                     alat,            &
                                     privect,         &
                                     ele_symbol,      & 
                                     ntype,           &
                                     natom,           &
                                     natomi,          & 
                                     ndynamic,        &
                                     ncoord,          &
                                     pripos,          &
                                     prifix,          &
                                     atom_type)
        
                        ! To write the POSCAR file
                        istrvalue=affopt(3)+affopt(4)*(affcount-1)
                        WRITE(outpos_name_tmp,'(A7,F8.4,A5)') 'AFFPOS_', istrvalue, '.vasp'

                        countnum=0
                        outpos_name='                              '
                        DO kk=1,LEN_TRIM(outpos_name_tmp)
                           IF ( outpos_name_tmp(kk:kk) .NE. ' ' ) THEN
                                   countnum=countnum+1
                                   outpos_name(countnum:countnum)=outpos_name_tmp(kk:kk)
                           ENDIF
                        ENDDO
                        outpos_name=TRIM(outpos_name)                        

                        CALL writepos(outpos_name,     &
                                      pos_title,       &
                                      alat,            &
                                      privect,         &
                                      ele_symbol,      & 
                                      ntype,           &
                                      natom,           &
                                      natomi,          & 
                                      ndynamic,        &
                                      ncoord,          &
                                      pripos,          &
                                      prifix)
        
                        outpos_name='AFFPOS0'
                        CALL writepos(outpos_name,     &
                                      pos_title,       &
                                      alat,            &
                                      privect,         &
                                      ele_symbol,      & 
                                      ntype,           &
                                      natom,           &
                                      natomi,          & 
                                      ndynamic,        &
                                      ncoord,          &
                                      pripos,          &
                                      prifix)
                ! ELSE
                ! 
                !         OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
                !         WRITE(19,*)
                !         WRITE(19,*) ' ----------------------------------------------------------------------------- '
                !         WRITE(19,*) '|                                     ERROR                                   |'
                !         WRITE(19,*) '|                            -----------------------                          |'
                !         WRITE(19,*) '|    Not find CONTCAR file as copying CONTCAR to AFFPOS0 and AFFPOS_          |'
                !         WRITE(19,*) '|    [istrvalue].vasp!                                                        |'
                !         WRITE(19,*) ' ----------------------------------------------------------------------------- '
                !         CLOSE(19)
                !         STOP
                ENDIF

        ENDIF

        ! Whether disturbing or Not

        CALL GET_COMMAND_ARGUMENT(8,option(8))
        IF ( option(8) .EQ. '-disturbing' ) THEN
                disturbing=1
                DO i=9,12
                   CALL GET_COMMAND_ARGUMENT(i,option(i))
                   READ(option(i),*) distopt(i-8)
                ENDDO
        ELSE
                disturbing=0
                DO i=1,4
                   distopt(i)=0.0000
                ENDDO
        ENDIF

        ! To stop as >= step_num
        IF ( affcount .GT. affopt(5) ) THEN
                ! CALL system ('')
                STOP
        ENDIF        
        ! To write the file LOOPCAR
        OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
        WRITE(87,*) ' AFFINE   ', TRIM(option(2)(2:7))
        WRITE(87,*) ' ---------------------------------------------------------- '
        WRITE(87,'(A17,F8.4)') '| TEN_STR      :', affopt(1)
        WRITE(87,'(A17,F8.4)') '| SHEAR_STR    :', affopt(2)
        WRITE(87,'(A17,F8.4)') '| ISTR0        :', affopt(3)
        WRITE(87,'(A17,F8.4)') '| STEP_LENGTH  :', affopt(4)
        WRITE(87,'(A17,I6)')   '| STEP_NUM     :', INT(affopt(5))
        WRITE(87,'(A17,I06.4)')   '| LOOP_NUM     :', affcount
        CLOSE(87)

        ! Affine deformation

        IF ( option(2).EQ.'-pure3d' ) THEN
!         IF ( option(2).EQ.'-pure3d' .OR. option(2).EQ.'-pure2d' ) THEN
                CALL pureshear(strain,disturbing,distopt,affcount,INT(affopt(5)))
         ELSEIF ( option(2).EQ.'-simp3d' ) THEN
!         ELSEIF ( option(2).EQ.'-simp3d' .OR. option(2).EQ.'-simp2d' ) THEN
                CALL simpleshear(strain,disturbing,distopt,affcount,INT(affopt(5)))
        ELSEIF ( option(2).EQ.'-unia2d' ) THEN
                CALL uniaxial2d(strain,disturbing,distopt,affcount,INT(affopt(5)))
        ELSEIF ( option(2).EQ.'-biax2d' ) THEN
                CALL biaxial2d(strain,disturbing,distopt,affcount,INT(affopt(5)))
        ENDIF

! ADAIS --alias -dxy3d/-dxy2d istartx iendx ispacingx istarty iendy ispacingy zvalue
! ADAIS --alias -dz istartz iendz ispacingz zvalue
ELSEIF ( option(1) .EQ. '--alias' ) THEN

        CALL GET_COMMAND_ARGUMENT(2,option(2))
        IF ( option(2).EQ.'-dxy3d' .OR. option(2).EQ.'-dxy2d' ) THEN

                DO i=1,6
                   CALL GET_COMMAND_ARGUMENT(i+2,option(i+2))
                   READ(option(i+2),*) aliopt(i)
                ENDDO                

                ! To read the zvalue
                CALL GET_COMMAND_ARGUMENT(9,option(9))
                IF ( LEN_TRIM(option(9)) .EQ. 0 ) THEN
                        zvalue=-1.0000
                ELSE
                        READ(option(9),*) zvalue
                ENDIF

                ! The number of step_num
                IF ( ABS(aliopt(2)-aliopt(1)) .LT. ZERO_TOLERANCE .OR.  &
                     (aliopt(2)-aliopt(1))*aliopt(3) .LT. 0.0000  .OR.  &
                     ABS(aliopt(3)) .LT. ZERO_TOLERANCE ) THEN
                        loopnumx=1
                ELSE
                        loopnumx=FLOOR((aliopt(2)-aliopt(1))/aliopt(3))+1
                ENDIF
                IF ( ABS(aliopt(5)-aliopt(4)) .LT. ZERO_TOLERANCE .OR.  &
                     (aliopt(5)-aliopt(4))*aliopt(6) .LT. 0.0000  .OR.  &
                     ABS(aliopt(6)) .LT. ZERO_TOLERANCE ) THEN
                        loopnumy=1
                ELSE
                        loopnumy=FLOOR((aliopt(5)-aliopt(4))/aliopt(6))+1
                ENDIF

                INQUIRE( FILE='LOOPCAR', EXIST=filefound )

                ! Strat from istri
                IF ( filefound ) THEN
                        OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
                        READ(87,*) titstr1, titstr2

                        IF ( TRIM(titstr1).EQ.'ALIAS' .AND. TRIM(titstr2).EQ.TRIM(option(2)(2:6)) ) THEN

                                READ(87,*) 
                                READ(87,*) tmpstr, tmpstr, tmpstr, istartx
                                READ(87,*) tmpstr, tmpstr, tmpstr, iendx
                                READ(87,*) tmpstr, tmpstr, tmpstr, ispacingx
                                READ(87,*) tmpstr, tmpstr, tmpstr, istarty
                                READ(87,*) tmpstr, tmpstr, tmpstr, iendy
                                READ(87,*) tmpstr, tmpstr, tmpstr, ispacingy
                                READ(87,*) tmpstr, tmpstr, tmpstr, zvalue0

                                IF( ABS(istartx  -aliopt(1)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(iendx    -aliopt(2)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(ispacingx-aliopt(3)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(istarty  -aliopt(4)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(iendy    -aliopt(5)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(ispacingy-aliopt(6)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(zvalue0  -zvalue   ) .LT. ZERO_TOLERANCE ) THEN
                
                                        READ(87,*) tmpstr, tmpstr, tmpstr, alicount
                                        alicount=alicount+1
        
                                ELSE
                                        alicount=1
                                ENDIF
                        ELSE
                                alicount=1
                        ENDIF
                        CLOSE(87)
                ! Start from istr0
                ELSE
                        alicount=1
                ENDIF
        
                IF ( alicount .EQ. 1 ) THEN
        
                        ! To define the dx and dy
                        dx=aliopt(1)
                        dy=aliopt(4)
                
                ELSEIF ( alicount .GT. 1 ) THEN
        
                        ! Whether CONTCAR file exists or Not
                        INQUIRE( FILE='CONTCAR', EXIST=filefound )
                        IF ( filefound ) THEN
                
                                ! To read the POSCAR file
                                inpos_name='CONTCAR'
                                CALL readpos(inpos_name,      &
                                             pos_title,       &
                                             alat,            &
                                             privect,         &
                                             ele_symbol,      & 
                                             ntype,           &
                                             natom,           &
                                             natomi,          & 
                                             ndynamic,        &
                                             ncoord,          &
                                             pripos,          &
                                             prifix,          &
                                             atom_type)
                
                                ! To write the POSCAR file
                                !! To define the dxr and dyr
                                dxr=FLOOR((alicount-2.0000)/loopnumy)*aliopt(3)+aliopt(1)
                                dyr=MOD(alicount-2,loopnumy)*aliopt(6)+aliopt(4)
                                WRITE(outpos_name_tmp,'(A7,F8.4,A1,F8.4,A5)') 'ALIPOS_', dxr,'_',dyr, '.vasp'

                                countnum=0
                                outpos_name='                              '
                                DO kk=1,LEN_TRIM(outpos_name_tmp)
                                   IF ( outpos_name_tmp(kk:kk) .NE. ' ' ) THEN
                                           countnum=countnum+1
                                           outpos_name(countnum:countnum)=outpos_name_tmp(kk:kk)
                                   ENDIF
                                ENDDO    
                                outpos_name=TRIM(outpos_name) 

                                CALL writepos(outpos_name,     &
                                              pos_title,       &
                                              alat,            &
                                              privect,         &
                                              ele_symbol,      & 
                                              ntype,           &
                                              natom,           &
                                              natomi,          & 
                                              ndynamic,        &
                                              ncoord,          &
                                              pripos,          &
                                              prifix)
                        ! ELSE
                        ! 
                        !         OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
                        !         WRITE(19,*)
                        !         WRITE(19,*) ' ----------------------------------------------------------------------------- '
                        !         WRITE(19,*) '|                                     ERROR                                   |'
                        !         WRITE(19,*) '|                            -----------------------                          |'
                        !         WRITE(19,*) '|    Not find CONTCAR file as copying CONTCAR to ALIPOS_[dx]_[dy].vasp        |'
                        !         WRITE(19,*) ' ----------------------------------------------------------------------------- '
                        !         CLOSE(19)
                        !         STOP
                        ENDIF

                        ! To define the dx and dy
                        dx=FLOOR((alicount-1.0000)/loopnumy)*aliopt(3)+aliopt(1)
                        dy=MOD(alicount-1,loopnumy)*aliopt(6)+aliopt(4)
        
                ENDIF

                ! To stop as >= step_num
                IF ( alicount .GT. loopnumx*loopnumy ) THEN
                        ! CALL system ('')
                        STOP
                ENDIF
                ! To write the file LOOPCAR
                OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
                WRITE(87,*) ' ALIAS   ', TRIM(option(2)(2:6))
                WRITE(87,*) ' ---------------------------------------------------------- '
                WRITE(87,'(A17,F8.4)') '| ISTARTX      :', aliopt(1)
                WRITE(87,'(A17,F8.4)') '| IENDX        :', aliopt(2)
                WRITE(87,'(A17,F8.4)') '| ISPACINGX    :', aliopt(3)
                WRITE(87,'(A17,F8.4)') '| ISTARTY      :', aliopt(4)
                WRITE(87,'(A17,F8.4)') '| IENDY        :', aliopt(5)
                WRITE(87,'(A17,F8.4)') '| ISPACINGY    :', aliopt(6)
                WRITE(87,'(A17,F8.4)') '| ZVALUE       :', zvalue
                WRITE(87,'(A17,I06.4)')   '| LOOP_NUM     :', alicount
                CLOSE(87)

                ! Alias deformation 
                IF ( option(2) .EQ. '-dxy3d' ) THEN
                        CALL alishear(dx,dy,zvalue,alicount,loopnumx*loopnumy)
                ELSEIF ( option(2) .EQ. '-dxy2d' ) THEN
                        CALL alishear2d(dx,dy,zvalue,alicount,loopnumx*loopnumy)
                ENDIF
        ELSEIF ( option(2) .EQ. '-dz' ) THEN

                DO i=1,3
                   CALL GET_COMMAND_ARGUMENT(i+2,option(i+2))
                   READ(option(i+2),*) aliopt(i)
                ENDDO

                ! To read the zvalue
                CALL GET_COMMAND_ARGUMENT(6,option(6))
                IF ( LEN_TRIM(option(6)) .EQ. 0 ) THEN
                        zvalue=-1.0000
                ELSE
                        READ(option(6),*) zvalue
                ENDIF

                ! The number of step_num
                IF ( ABS(aliopt(2)-aliopt(1)) .LT. ZERO_TOLERANCE .OR.  &
                     (aliopt(2)-aliopt(1))*aliopt(3) .LT. 0.0000  .OR.  &
                     ABS(aliopt(3)) .LT. ZERO_TOLERANCE ) THEN
                        loopnumz=1
                ELSE
                        loopnumz=FLOOR((aliopt(2)-aliopt(1))/aliopt(3))+1
                ENDIF

                INQUIRE( FILE='LOOPCAR', EXIST=filefound )

                ! Strat from istri
                IF ( filefound ) THEN
                        OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
                        READ(87,*) titstr1, titstr2

                        IF ( TRIM(titstr1).EQ.'ALIAS' .AND. TRIM(titstr2).EQ.TRIM(option(2)(2:6)) ) THEN

                                READ(87,*) 
                                READ(87,*) tmpstr, tmpstr, tmpstr, istartz
                                READ(87,*) tmpstr, tmpstr, tmpstr, iendz
                                READ(87,*) tmpstr, tmpstr, tmpstr, ispacingz
                                READ(87,*) tmpstr, tmpstr, tmpstr, zvalue0

                                IF( ABS(istartz  -aliopt(1)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(iendz    -aliopt(2)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(ispacingz-aliopt(3)) .LT. ZERO_TOLERANCE .AND. &
                                    ABS(zvalue0  -zvalue   ) .LT. ZERO_TOLERANCE ) THEN
                
                                        READ(87,*) tmpstr, tmpstr, tmpstr, alicount
                                        alicount=alicount+1
        
                                ELSE
                                        alicount=1
                                ENDIF
                        ELSE
                                alicount=1
                        ENDIF
                        CLOSE(87)
                ! Start from istr0
                ELSE
                        alicount=1
                ENDIF

                IF ( alicount .EQ. 1 ) THEN
        
                        ! To define the dz
                        dz=aliopt(1)
                
                ELSEIF ( alicount .GT. 1) THEN
        
                        ! Whether CONTCAR file exists or Not
                        INQUIRE( FILE='CONTCAR', EXIST=filefound )
                        IF ( filefound ) THEN
                
                                ! To read the POSCAR file
                                inpos_name='CONTCAR'
                                CALL readpos(inpos_name,      &
                                             pos_title,       &
                                             alat,            &
                                             privect,         &
                                             ele_symbol,      & 
                                             ntype,           &
                                             natom,           &
                                             natomi,          & 
                                             ndynamic,        &
                                             ncoord,          &
                                             pripos,          &
                                             prifix,          &
                                             atom_type)
                
                                ! To write the POSCAR file
                                !! To define the dzr
                                dzr=(alicount-2)*aliopt(3)+aliopt(1)
                                WRITE(outpos_name_tmp,'(A7,F8.4,A5)') 'ALIPOS_', dzr, '.vasp'

                                countnum=0
                                outpos_name='                              '
                                DO kk=1,LEN_TRIM(outpos_name_tmp)
                                   IF ( outpos_name_tmp(kk:kk) .NE. ' ' ) THEN
                                           countnum=countnum+1
                                           outpos_name(countnum:countnum)=outpos_name_tmp(kk:kk)
                                   ENDIF
                                ENDDO                                 
                                outpos_name=TRIM(outpos_name)

                                CALL writepos(outpos_name,     &
                                              pos_title,       &
                                              alat,            &
                                              privect,         &
                                              ele_symbol,      & 
                                              ntype,           &
                                              natom,           &
                                              natomi,          & 
                                              ndynamic,        &
                                              ncoord,          &
                                              pripos,          &
                                              prifix)
                        ! ELSE
                        ! 
                        !         OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
                        !         WRITE(19,*)
                        !         WRITE(19,*) ' ----------------------------------------------------------------------------- '
                        !         WRITE(19,*) '|                                     ERROR                                   |'
                        !         WRITE(19,*) '|                            -----------------------                          |'
                        !         WRITE(19,*) '|    Not find CONTCAR file as copying CONTCAR to ALIPOS_[dz].vasp             |'
                        !         WRITE(19,*) ' ----------------------------------------------------------------------------- '
                        !         CLOSE(19)
                        !         STOP                                
                        ENDIF

                        ! To define the dz
                        dz=(alicount-1)*aliopt(3)+aliopt(1)
        
                ENDIF

                ! To stop as >= step_num
                IF ( alicount .GT. loopnumz ) THEN
                        ! CALL system ('')
                        STOP
                ENDIF
                ! To write the file LOOPCAR
                OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
                WRITE(87,*) ' ALIAS   ', TRIM(option(2)(2:6))
                WRITE(87,*) ' ---------------------------------------------------------- '
                WRITE(87,'(A17,F8.4)') '| ISTARTZ      :', aliopt(1)
                WRITE(87,'(A17,F8.4)') '| IENDZ        :', aliopt(2)
                WRITE(87,'(A17,F8.4)') '| ISPACINGZ    :', aliopt(3)
                WRITE(87,'(A17,F8.4)') '| ZVALUE       :', zvalue
                WRITE(87,'(A17,I06.4)')   '| LOOP_NUM     :', alicount
                CLOSE(87)                

                ! Alias deformation
                CALL adhesion(dz,zvalue,alicount,loopnumz)
        ENDIF

! ADAIS --proj -mat mat11 mat12 mat13 mat21 mat22 mat23 mat31 mat32 mat33
! ADAIS --proj -rot angx angy angz
! ADAIS --proj -ind pvh pvk pvl uvu uvv uvw
ELSEIF ( option(1) .EQ. '--proj' ) THEN

        CALL GET_COMMAND_ARGUMENT(2,option(2))
        IF ( option(2) .EQ. '-mat' ) THEN
                DO i=1,3
                   DO j=1,3
                      CALL GET_COMMAND_ARGUMENT(((i-1)*3+j+2),option((i-1)*3+j+2))
                      READ(option((i-1)*3+j+2),*) matopt(i,j)
                   ENDDO
                ENDDO
                CALL matproj(matopt)
        ELSEIF ( option(2) .EQ. '-rot' ) THEN
                DO i=1,3
                   CALL GET_COMMAND_ARGUMENT(i+2,option(i+2))
                   READ(option(i+2),*) rotopt(i)
                ENDDO
                CALL rotproj(rotopt)
        ELSEIF ( option(2) .EQ. '-ind' ) THEN
                DO i=1,6
                   CALL GET_COMMAND_ARGUMENT(i+2,option(i+2))
                   READ(option(i+2),*) indxopt(i)
                ENDDO
                CALL indxproj(indxopt(1),indxopt(2),indxopt(3),  &
                              indxopt(4),indxopt(5),indxopt(6))
        ENDIF

! ADAIS --newlat vec11 vec12 vec13 vec21 vec22 vec23 vec31 vec32 vec33
ELSEIF ( option(1) .EQ. '--newlat' ) THEN

        DO i=1,3
           DO j=1,3
              CALL GET_COMMAND_ARGUMENT(((i-1)*3+j+1),option((i-1)*3+j+1))
              READ(option((i-1)*3+j+1),*) newlat(i,j)
           ENDDO
        ENDDO
        CALL redefinelat(newlat)

! ADAIS --super sup_1 sup_2 sup_3
ELSEIF ( option(1) .EQ. '--super' ) THEN

        DO i=1,3
           CALL GET_COMMAND_ARGUMENT(i+1,option(i+1))
        ENDDO
        IF ( LEN_TRIM(option(2)).NE.0 .AND.  &
             LEN_TRIM(option(3)).NE.0 .AND.  &
             LEN_TRIM(option(4)).NE.0 ) THEN
                
                DO i=1,3
                        READ(option(i+1),*) sup_range(i)
                ENDDO
        ENDIF

        inpos_name='INPOS'
        CALL supercell(inpos_name,sup_range)

! ! ADAIS --twin 1/2 [layer_spac]
! ELSEIF ( option(1) .EQ. '--twin' ) THEN
! 
!         CALL GET_COMMAND_ARGUMENT(2,option(2))
!         IF ( option(2) .EQ. '1' ) THEN
!                 CALL GET_COMMAND_ARGUMENT(3,option(3))
!                 READ(option(3),*) layer_spac
!         ELSEIF ( option(2) .EQ. '2' ) THEN
!                 layer_spac=0.0000
!         ENDIF
!         READ(option(2),*) twintype
! 
!         CALL twinbuild(twintype, layer_spac)
! 
! ! ADAIS --disp
! ELSEIF ( option(1) .EQ. '--disp' ) THEN
!         CALL strdisp
! 
! ADAIS --ieee -3d/-2d
ELSEIF ( option(1) .EQ. '--ieee' ) THEN

        CALL GET_COMMAND_ARGUMENT(2,option(2))
        IF ( option(2) .EQ. '-3d' ) THEN
                CALL recell
        ELSEIF ( option(2) .EQ. '-2d' ) THEN
                CALL recell2d
        ENDIF

! ADAIS --symm [symprec]
ELSEIF ( option(1) .EQ. '--symm' ) THEN

        CALL GET_COMMAND_ARGUMENT(2,option(2))
        IF ( LEN_TRIM(option(2)) .EQ. 0 ) THEN
                sympreci=-1.0000
        ELSE
                READ(option(2),*) sympreci
        ENDIF

        CALL symmetry(sympreci)

! ADAIS --output
ELSEIF ( option(1) .EQ. '--output' ) THEN
        CALL writeout

! ADAIS --bond
ELSEIF ( option(1) .EQ. '--bond' ) THEN

        CALL GET_COMMAND_ARGUMENT(2,option(2))
        READ(option(2),*) bondcut

        CALL strdisp
        CALL writebond(bondcut)

! ADAIS --fconv
ELSEIF ( option(1) .EQ. '--fconv' ) THEN
        CALL forceconv
        
! ADAIS --incar -rlx/-stc [pressure]
ELSEIF ( option(1) .EQ. '--incar' ) THEN
        DO i=2,3
           CALL GET_COMMAND_ARGUMENT(i,option(i))
        ENDDO
        ! The default value of "pressure" is 0000
        IF ( LEN_TRIM(option(3)) .EQ. 0 ) THEN
                pressure=0000
        ELSE 
                READ(option(3),*) pressure
        ENDIF
        ! Unit "GPa" --> "KBar"
        pressure=pressure*10.0000

        IF ( option(2) .EQ. '-rlx' ) THEN
                CALL incarx('rlx',pressure)
        ELSEIF ( option(2) .EQ. '-atm' ) THEN
                CALL incarx('atm',pressure)
        ELSEIF ( option(2) .EQ. '-stc' ) THEN
                CALL incarx('stc',pressure)
        ENDIF

! ADAIS --kpt -kmesh k_1 k_2 k_3 [kscheme]
! ADAIS --kpt -kppra kppra [kscheme]
! ADAIS --kpt -kspac kspacing [kscheme]
ELSEIF ( option(1) .EQ. '--kpt' ) THEN

        CALL GET_COMMAND_ARGUMENT(2,option(2))
        IF ( option(2) .EQ. '-kmesh' ) THEN

                DO i=1,3
                   CALL GET_COMMAND_ARGUMENT(i+2,option(i+2))
                   READ(option(i+2),*) kmesh(i)
                ENDDO
                CALL GET_COMMAND_ARGUMENT(6,kscheme)
                ! The default value of "kscheme" is Gamma
                IF ( LEN_TRIM(kscheme) .EQ. 0 ) THEN
                        kscheme='Gamma'
                ENDIF
                CALL kpt(kmesh, kscheme(1:1))

        ELSEIF ( option(2) .EQ. '-kppra' ) THEN

                CALL GET_COMMAND_ARGUMENT(3,option(3))
                ! The default value of "kppra" is 1000
                IF ( LEN_TRIM(option(3)) .EQ. 0 ) THEN
                        kppra=1000
                ELSE
                        READ(option(3),*) kppra
                ENDIF
                CALL GET_COMMAND_ARGUMENT(4,kscheme)
                ! The default value of "kscheme" is Gamma
                IF ( LEN_TRIM(kscheme) .EQ. 0 ) THEN
                          kscheme='Gamma'
                ENDIF
                CALL kpta(kppra, kscheme(1:1))

        ELSEIF ( option(2) .EQ. '-kspac' ) THEN

                CALL GET_COMMAND_ARGUMENT(3,option(3))
                ! The default value of "kspac" is 0.5
                IF ( LEN_TRIM(option(3)) .EQ. 0 ) THEN
                        kspac=0.5
                ELSE
                        READ(option(3),*) kspac
                ENDIF
                CALL GET_COMMAND_ARGUMENT(4,kscheme)
                ! The default value of "kscheme" is Gamma
                IF ( LEN_TRIM(kscheme) .EQ. 0 ) THEN
                          kscheme='Gamma'
                ENDIF
                CALL kptv(kspac, kscheme(1:1))                

        ENDIF               

! ADAIS --clean
ELSEIF ( option(1) .EQ. '--clean' ) THEN
        CALL clean      
! ADAIS --help
ELSEIF ( option(1) .EQ. '--help' ) THEN
        CALL help
ELSE       
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Not such input option! For more information, one may use: ADAIS --help.  |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP        
ENDIF

END PROGRAM main
