SUBROUTINE matproj(projmat)
!
!  Purpose:
!
!    Generalized method of projection
!
!    To project one specific orientation (i.e., the imposed strain direction) of a given crystal
!    to be parallel to one axis vector (e.g. y) for affine tensile deformation, and projects 
!    one axis vector (e.g. x) to be perpendicular to the slip plane and another one (e.g. y) to
!    be parallel to the slip direction in that plane for the affine shearing deformation
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    projmat   : The projection matrix, i.e., privect=indxvect*projmat
!    projmatI  : The inverse matrix of projmat, i.e., indxvect=privect*projmatI
!    matvect   : The vectors defining the unit cell of the system after projection
!
!  Note:
!    To operate the lattice vectors by matrix
!    ---------------------------------------------
!         |a11 a12 a13|       |mat11 mat12 mat13|
!    R=   |a21 a22 a23|  MAT= |mat21 mat22 mat23|
!         |a31 a32 a33|       |mat31 mat32 mat33|
!    ---------------------------------------------
!         |aa11 aa12 aa13|
!    RR=  |aa21 aa22 aa23|
!         |aa31 aa32 aa33|
!    ---------------------------------------------
!    RR= R*MAT
!    Notice: aa11=a11*mat11+a12*mat21+a13*mat31
!            aa12=a11*mat12+a12*mat22+a13*mat32
!            aa13=a11*mat13+a12*mat23+a13*mat33
!    Notice: aa21=a21*mat11+a22*mat21+a23*mat31
!            aa22=a21*mat12+a22*mat22+a23*mat32
!            aa23=a21*mat13+a22*mat23+a23*mat33
!    Notice: aa31=a31*mat11+a32*mat21+a33*mat31
!            aa32=a31*mat12+a32*mat22+a33*mat32
!            aa33=a31*mat13+a32*mat23+a33*mat33
!    ---------------------------------------------
!    In VESTA, orientation is the inverse of MAT
!    e.g. the 1st row corresponds to mat11 mat21 mat31
!         the 1st col corresponds to mat11 mat12 mat13 
!    ---------------------------------------------
!
!    For "projmat":  privect=indxvect*projmat
!    For "projmatI": indxvect=privect*projmatI 
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

DOUBLE PRECISION :: projmat(3,3), projmatI(3,3)
DOUBLE PRECISION :: matvect(3,3)

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

! Read the transform matrix from the file 'PROJMAT'
! meanwhile the matrix is transposed

! OPEN(20,FILE='PROJMAT')
! READ(20,*) title_projmat
! DO i=1,3
!    READ(20,*) (projmatI(j,i),j=1,3) 
! ENDDO
! CLOSE(20)

DO i=1,3
   DO j=1,3
      projmatI(i,j)=projmat(j,i)
   ENDDO
ENDDO

! Transform the cell vector to new vector under strain
!        matvect(i,j)=privect(i,j)(I+projmatI(i,j))

DO i=1,3
   DO j=1,3
      matvect(i,j)=0.0000
      DO k=1,3
         matvect(i,j)=matvect(i,j)+privect(i,k)*projmatI(k,j)
      ENDDO
   ENDDO 
ENDDO

! To output the results

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
WRITE(19,*)
WRITE(19,*) ' CRYSTAL PROJECTION (MATPROJ)'
WRITE(19,*) ' ----------------------------------------------------------------------------- '
WRITE(19,*) '| Before Projction, the three lattice vectors:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO

WRITE(19,*) '| '
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| Orientation matrix is as follows:'
DO i=1,3
   WRITE(19,"(A11,3F12.6)") '|         ', (projmat(i,j),j=1,3)
ENDDO
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| '

WRITE(19,*) '| After Projction, the three lattice vectors:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (matvect(i,j), j=1,3)
ENDDO

CLOSE(19)

! To write the POSCAR file

outpos_name='PROJPOS'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              matvect,         &
              ele_symbol,      & 
              ntype,           &
              natom,           &
              natomi,          & 
              ndynamic,        &
              ncoord,          &
              pripos,          &
              prifix)
            
! End of writing the POSCAR file

END SUBROUTINE matproj
