!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
!
!                         Standardized Unit Cell
! 
!        >>>>>>>>>>>>>>>>>>  Triclinic lattice  <<<<<<<<<<<<<<<<<<      
!
!               
!      
!        Niggli reduced cell is used for choosing a, b, c
!        
!        alpha > 90 degree, bata > 90 degree 
!             
!        c < a < b
!              
!        c is set along +z direction of Cartesian coordinates
!        a is set in x-z plane of Cartesian coordinates so that
!          c*a is along +y direction of Cartesian coordinates
!
!        >>>>>>>>>>>>>>>>>>  Monoclinic lattice  <<<<<<<<<<<<<<<<<<
!      
!        b axis is taken as the unique axis
!      
!        alpha = 90 degree and gamma = 90 degree
!        bata > 90 degree
! 
!        c < a        
!
!        a is set in x-z plane of Cartesian coordinates      
!        b is set along +y direction of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!        
!
!        >>>>>>>>>>>>>>>>>  Orthorhombic lattice  <<<<<<<<<<<<<<<<<      
!
!      
!        alpha = beta = gamma = 90 degree
!
!        c < a < b
!      
!        a is set along +x direction of Cartesian coordinates
!        b is set along +y direction of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!
!
!        >>>>>>>>>>>>>>>>>>  Tetragonal lattice  <<<<<<<<<<<<<<<<<<      
!      
!
!        alpha = beta = gamma = 90 degree
!      
!        a = b
!      
!        a is set along +x direction of Cartesian coordinates
!        b is set along +y direction of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!
!      
!        >>>>>>>>>>>>>>>>>>  Rhombohedral lattice  <<<<<<<<<<<<<<<<<<      
!
!      
!        alpha = beta = 90 degree
!        gamma = 120 degree
!      
!        a = b
!      
!        a is set along +x direction of Cartesian coordinates
!        b is set in x-y plane of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!
!        
!        >>>>>>>>>>>>>>>>>>   Hexagonal lattice   <<<<<<<<<<<<<<<<<<      
!
!      
!        alpha = beta = 90 degree
!        gamma = 120 degree
!      
!        a = b
!      
!        a is set along +x direction of Cartesian coordinates
!        b is set in x-y plane of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!
!
!        >>>>>>>>>>>>>>>>>>     Cubic lattice     <<<<<<<<<<<<<<<<<<      
!
!      
!        alpha = beta = gamma = 90 degree
!      
!        a = b = c
!     
!        a is set along +x direction of Cartesian coordinates
!        b is set along +y direction of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!      
!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

! MODULE defs_basis
! IMPLICIT NONE
! INTEGER, PARAMETER :: dp=KIND(1.00D0)
! END MODULE defs_basis

SUBROUTINE recell
!
!  Purpose:
!
!    To transform the structure to IEEE-format for 3D materials
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    space_group: The space group number
!    refine_cell:
!    lattice   : The vectors defining the unit cell of the system with lattice(i,j)=privect(i,j)
!    positions : Three coordinates for each atom with positions(j,i)=pripos(i,j)
!    pricoord  : Direct or Cartesian
!
!    lega(b,c) : The length of privect(i,*)
!    lega(b,c)r: The length of recipvect(i,*)
!    cos_alpha : The angle between privect(2,*) and privect(3,*)
!    cos_bata  : The angle between privect(3,*) and privect(1,*)
!    cos_gamma : The angle between privect(1,*) and privect(2,*)
!    cos_alphar: The angle between recipvect(2,*) and recipvect(3,*)
!    cos_batar : The angle between recipvect(3,*) and recipvect(1,*)
!    cos_gammar: The angle between recipvect(1,*) and recipvect(2,*)
!
!    recipvect : The reciprocal vector of redvect
!
!
!
!

USE defs_basis
USE spglib_f08
USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk, m, nn
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: space_group, refine_cell

DOUBLE PRECISION :: lattice(3,3), positions(3,ATOMMAX)
DOUBLE PRECISION :: lega, legb, legc
DOUBLE PRECISION :: legar, legbr, legcr
DOUBLE PRECISION :: cos_alpha, cos_bata, cos_gamma
DOUBLE PRECISION :: cos_alphar, cos_batar, cos_gammar
DOUBLE PRECISION :: cos_angle
DOUBLE PRECISION :: temp, temppos(ATOMMAX,3), tempvect(5,5), recipvect(3,3)

CHARACTER(len=20) :: pricoord, symbol

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! Output the initial structure

IF ( ncoord .EQ. 0 ) THEN
        pricoord='Direct'
ELSEIF ( ncoord .EQ. 1 ) THEN
        pricoord='Cartesian'
ENDIF

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*)
WRITE(19,*) ' RECELL STRUCTURE (3D)'
WRITE(19,*) ' ----------------------------------------------------------------------------- '

WRITE(19,*) '| The initial structure:'
WRITE(19,*) '| '

WRITE(19,'(A3,F8.5)') '| ', alat
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ', (ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ', (natomi(i), i=1,ntype)
WRITE(19,'(A3,A9)') '| ', pricoord
DO i=1, natom
   WRITE(19,"(A3,I4,3F20.10)") '| ', atom_type(i), (pripos(i,j),j=1,3)
ENDDO

CLOSE(19)      

! End of outputing the initial structure 

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

! To get the space group of the structure "INPOS" 

DO i=1,3
   DO j=1,3
      lattice(i,j)=privect(i,j)
   ENDDO
ENDDO

DO i=1,natom
   DO j=1,3
      positions(j,i)=pripos(i,j)
   ENDDO
ENDDO

symbol='             '
space_group=spg_get_international(symbol,lattice,positions,atom_type,natom,SYMPREC)

IF ( space_group .EQ. 0 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Cannot get space group!                                                  |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP
ENDIF

! To refine the new cell with standardized unit cell

refine_cell=spg_refine_cell(lattice,positions,atom_type,natom,SYMPREC)

IF ( refine_cell .EQ. 0 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Cannot recell the structure!                                             |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP
ENDIF

DO i=1,3
   DO j=1,3
      privect(i,j)=lattice(i,j)
   ENDDO
ENDDO

!! refine_cell is the atom number in the recelled structure

natom=refine_cell

DO i=1, ntype
   natomi(i)=0
   DO j=1, natom
      IF ( atom_type(j) .EQ. i ) THEN
              natomi(i)=natomi(i)+1
      ENDIF
   ENDDO
ENDDO

m=1
DO i=1,ntype
   DO j=1,natom
      IF ( atom_type(j) .EQ. i ) THEN
              DO k=1,3
                 pripos(m,k)=positions(k,j)
              ENDDO
              m=m+1 
      ENDIF
   ENDDO
ENDDO

! Othorhombic lattice: c < a < b

IF ( space_group.GE.16 .AND. space_group.LE.74 ) THEN

        DO i=1,2                
           IF ( privect(3,3) .GT. privect(i,i) ) THEN
                   temp=privect(3,3)
                   privect(3,3)=privect(i,i)
                   privect(i,i)=temp

                   DO j=1,natom
                      temp=pripos(j,3)
                      pripos(j,3)=pripos(j,i)
                      pripos(j,i)=temp
                   ENDDO
           ENDIF
        ENDDO

        IF ( privect(1,1) .GT. privect(2,2) ) THEN
                temp=privect(1,1)
                privect(1,1)=privect(2,2)
                privect(2,2)=temp

                DO j=1,natom
                   temp=pripos(j,1)
                   pripos(j,1)=pripos(j,2)
                   pripos(j,2)=temp
                ENDDO
        ENDIF            

ENDIF

! Monoclinic lattice: c < a

IF ( space_group.GE.3 .AND. space_group.LE.15 ) THEN

        lega=SQRT(privect(1,1)**2+privect(1,2)**2+privect(1,3)**2)
        legc=SQRT(privect(3,1)**2+privect(3,2)**2+privect(3,3)**2)

        IF ( lega .LT. legc ) THEN
               temp=privect(1,1)
               privect(1,1)=privect(3,3)
               privect(3,3)=temp
               
               privect(1,3)=privect(3,1)
               privect(3,1)=0.0000

               DO j=1,natom
                   temp=pripos(j,1)
                   pripos(j,1)=pripos(j,3)
                   pripos(j,3)=temp
               ENDDO
       ENDIF
        
       IF ( lega .GT. legc ) THEN
               temp=privect(3,3)
               privect(3,3)=legc

               privect(1,1)=lega*temp/legc
               privect(1,3)=lega*privect(3,1)/legc
               privect(3,1)=0.0000
       ENDIF

ENDIF

! Triclinic lattice: c < a < b 

IF ( space_group.GE.1 .AND. space_group.LE.2 ) THEN
        
        lega=SQRT(privect(1,1)**2+privect(1,2)**2+privect(1,3)**2)
        legb=SQRT(privect(2,1)**2+privect(2,2)**2+privect(2,3)**2)
        legc=SQRT(privect(3,1)**2+privect(3,2)**2+privect(3,3)**2)

        cos_alpha=(privect(2,1)*privect(3,1)+privect(2,2)    &
                  *privect(3,2)+privect(2,3)*privect(3,3))/legb/legc
        cos_bata=(privect(1,1)*privect(3,1)+privect(1,2)     &
                  *privect(3,2)+privect(1,3)*privect(3,3))/lega/legc
        cos_gamma=(privect(1,1)*privect(2,1)+privect(1,2)    &
                  *privect(2,2)+privect(1,3)*privect(2,3))/lega/legb

!! define the reciprocal matrix of privect

        DO i=1,3
           DO j=1,3
              tempvect(i,j)=privect(i,j)
           ENDDO
        ENDDO

        DO i=1,3
           DO k=4,5
              tempvect(i,k)=tempvect(i,k-3)
           ENDDO
        ENDDO

        DO i=4,5
           DO k=1,5
              tempvect(i,k)=tempvect(i-3,k)
           ENDDO
        ENDDO
        
        DO i=1,3
           DO k=1,3
              recipvect(i,k)=tempvect(i+1,k+1)*tempvect(i+2,k+2)  &
                            -tempvect(i+1,k+2)*tempvect(i+2,k+1)
           ENDDO
        ENDDO

!! end of defining the reciprocal matrix of privect


        legar=SQRT(recipvect(1,1)**2+recipvect(1,2)**2+recipvect(1,3)**2)
        legbr=SQRT(recipvect(2,1)**2+recipvect(2,2)**2+recipvect(2,3)**2)
        legcr=SQRT(recipvect(3,1)**2+recipvect(3,2)**2+recipvect(3,3)**2)

        cos_alphar=(privect(1,1)*recipvect(1,1)+privect(1,2)    &
                   *recipvect(1,2)+privect(1,3)*recipvect(1,3))/lega/legar
        cos_batar=(privect(2,1)*recipvect(2,1)+privect(2,2)     &
                   *recipvect(2,2)+privect(2,3)*recipvect(2,3))/legb/legbr
        cos_gammar=(privect(3,1)*recipvect(3,1)+privect(3,2)    &
                   *recipvect(3,2)+privect(3,3)*recipvect(3,3))/legc/legcr

        DO i=1,3
           DO j=1,3
              tempvect(i,j)=0.0000
           ENDDO
        ENDDO

!!!     a <= b <= c           

        IF ( lega.LE.legc .AND. legb.LE.legc .AND. lega.LE.legb ) THEN
               tempvect(3,3)=lega

               tempvect(1,3)=legb*cos_gamma
               tempvect(1,1)=SQRT(legb**2-tempvect(1,3)**2)

               tempvect(2,3)=legc*cos_bata
               tempvect(2,2)=legc*cos_gammar
               tempvect(2,1)=SQRT(legc**2-tempvect(2,3)**2-tempvect(2,2)**2)

!!!            To make the Angle unchanged

               cos_angle=(tempvect(1,1)*tempvect(2,1)  &
                         +tempvect(1,2)*tempvect(2,2)  &
                         +tempvect(1,3)*tempvect(2,3)) &
                         /legb/legc

               IF ( ABS(cos_angle-cos_alpha) .GT. ZERO_TOLERANCE ) THEN
                       tempvect(2,1)=-tempvect(2,1)
               ENDIF

               DO j=1,natom
                  temppos(j,3)=pripos(j,1)
                  temppos(j,1)=pripos(j,2)
                  temppos(j,2)=pripos(j,3)
               ENDDO
        ENDIF

!!!     a <= c <= b               

        IF ( lega.LE.legb .AND. legc.LE.legb .AND. lega.LE.legc ) THEN
               tempvect(3,3)=lega

               tempvect(1,3)=legc*cos_bata
               tempvect(1,1)=SQRT(legc**2-tempvect(1,3)**2)

               tempvect(2,3)=legb*cos_gamma
               tempvect(2,2)=legb*cos_batar
               tempvect(2,1)=SQRT(legb**2-tempvect(2,3)**2-tempvect(2,2)**2)

!!!            To make the Angle unchanged

               cos_angle=(tempvect(1,1)*tempvect(2,1)  &
                         +tempvect(1,2)*tempvect(2,2)  &
                         +tempvect(1,3)*tempvect(2,3)) &
                         /legb/legc

               IF ( ABS(cos_angle-cos_alpha) .GT. ZERO_TOLERANCE ) THEN
                       tempvect(2,1)=-tempvect(2,1)
               ENDIF

               DO j=1,natom
                  temppos(j,3)=pripos(j,1)
                  temppos(j,1)=pripos(j,3)
                  temppos(j,2)=pripos(j,2)
               ENDDO
        ENDIF

!!!     b <= a <= c

        IF ( legb.LE.legc .AND. lega.LE.legc .AND. legb.LE.lega ) THEN
               tempvect(3,3)=legb

               tempvect(1,3)=lega*cos_gamma
               tempvect(1,1)=SQRT(lega**2-tempvect(1,3)**2)

               tempvect(2,3)=legc*cos_alpha
               tempvect(2,2)=legc*cos_gammar
               tempvect(2,1)=SQRT(legc**2-tempvect(2,3)**2-tempvect(2,2)**2)

!!!            To make the Angle unchanged                       

               cos_angle=(tempvect(1,1)*tempvect(2,1)  &
                         +tempvect(1,2)*tempvect(2,2)  &
                         +tempvect(1,3)*tempvect(2,3)) &
                         /lega/legc

               IF ( ABS(cos_angle-cos_bata) .GT. ZERO_TOLERANCE ) THEN
                       tempvect(2,1)=-tempvect(2,1)
               ENDIF

               DO j=1,natom
                  temppos(j,3)=pripos(j,2)
                  temppos(j,1)=pripos(j,1)
                  temppos(j,2)=pripos(j,3)
               ENDDO
        ENDIF

!!!     b <= c <= a

        IF ( legb.LE.lega .AND. legc.LE.lega .AND. legb.LE.legc ) THEN
               tempvect(3,3)=legb

               tempvect(1,3)=legc*cos_alpha
               tempvect(1,1)=SQRT(legc**2-tempvect(1,3)**2)

               tempvect(2,3)=lega*cos_gamma
               tempvect(2,2)=lega*cos_alphar
               tempvect(2,1)=SQRT(lega**2-tempvect(2,3)**2-tempvect(2,2)**2)

!!!            To make the Angle unchanged

               cos_angle=(tempvect(1,1)*tempvect(2,1)  &
                         +tempvect(1,2)*tempvect(2,2)  &
                         +tempvect(1,3)*tempvect(2,3)) &
                         /lega/legc

               IF ( ABS(cos_angle-cos_bata) .GT. ZERO_TOLERANCE ) THEN
                       tempvect(2,1)=-tempvect(2,1)
               ENDIF


               DO j=1,natom
                  temppos(j,3)=pripos(j,2)
                  temppos(j,1)=pripos(j,3)
                  temppos(j,2)=pripos(j,1)
               ENDDO
        ENDIF

!!!     c <= a <= b

        IF ( legc.LE.legb .AND. lega.LE.legb .AND. legc.LE.lega ) THEN
               tempvect(3,3)=legc

               tempvect(1,3)=lega*cos_bata
               tempvect(1,1)=SQRT(lega**2-tempvect(1,3)**2)

               tempvect(2,3)=legb*cos_alpha
               tempvect(2,2)=legb*cos_batar
               tempvect(2,1)=SQRT(legb**2-tempvect(2,3)**2-tempvect(2,2)**2)

!!!            To make the Angle unchanged                       

               cos_angle=(tempvect(1,1)*tempvect(2,1)  &
                         +tempvect(1,2)*tempvect(2,2)  &
                         +tempvect(1,3)*tempvect(2,3)) &
                         /lega/legb

               IF ( ABS(cos_angle-cos_gamma) .GT. ZERO_TOLERANCE ) THEN
                       tempvect(2,1)=-tempvect(2,1)
               ENDIF

               DO j=1,natom
                  temppos(j,3)=pripos(j,3)
                  temppos(j,1)=pripos(j,1)
                  temppos(j,2)=pripos(j,2)
               ENDDO
        ENDIF

!!!     c <= b <= a

        IF ( legc.LE.lega .AND. legb.LE.lega .AND. legc.LE.legb ) THEN
               tempvect(3,3)=legc

               tempvect(1,3)=legb*cos_alpha
               tempvect(1,1)=SQRT(legb**2-tempvect(1,3)**2)

               tempvect(2,3)=lega*cos_bata
               tempvect(2,2)=lega*cos_alphar
               tempvect(2,1)=SQRT(lega**2-tempvect(2,3)**2-tempvect(2,2)**2)

!!!            To make the Angle unchanged

               cos_angle=(tempvect(1,1)*tempvect(2,1)  &
                         +tempvect(1,2)*tempvect(2,2)  &
                         +tempvect(1,3)*tempvect(2,3)) &
                         /lega/legb

               IF ( ABS(cos_angle-cos_gamma) .GT. ZERO_TOLERANCE ) THEN
                       tempvect(2,1)=-tempvect(2,1)
               ENDIF

               DO j=1,natom
                  temppos(j,3)=pripos(j,3)
                  temppos(j,1)=pripos(j,2)
                  temppos(j,2)=pripos(j,1)
               ENDDO
        ENDIF             

        DO i=1,3
           DO j=1,3
              privect(i,j)=tempvect(i,j)
           ENDDO
        ENDDO

        DO i=1,natom
           DO j=1,3
              pripos(i,j)=temppos(i,j)
           ENDDO
        ENDDO

!!!     alpha > 90 degree, bata > 90 degree

        IF ( privect(1,3) .GT. 0 ) THEN
                privect(1,1)=-privect(1,1)
                privect(1,3)=-privect(1,3)
       
                DO i=1,natom
                   pripos(i,1)=1-pripos(i,1)
                ENDDO
        ENDIF
        IF ( privect(2,3) .GT. 0 ) THEN
                DO i=1,3
                   privect(2,i)=-privect(2,i)
                ENDDO

                DO i=1,natom
                   pripos(i,2)=1-pripos(i,2)
                ENDDO
        ENDIF

ENDIF

! Create the POSCAR for first-principles calculation

!! output the recelled structure

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*) '| '
WRITE(19,*) '| The recelled structure:'
WRITE(19,*) '| '

WRITE(19,'(A3,F8.5)') '| ',alat
DO i=1,3
   WRITE(19,"(A3,3F20.10)") '| ', (privect(i,j),j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ', (ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ', (natomi(i), i=1,ntype)
WRITE(19,'(A3,A9)') '| ', pricoord
DO i=1, natom
   WRITE(19,"(A3,I4,3F20.10)") '| ', atom_type(i), (pripos(i,j),j=1,3)
ENDDO

CLOSE(19)

!! end of outputing the recelled structure   

! To write the POSCAR file

outpos_name='IEEEPOS'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              privect,         &
              ele_symbol,      & 
              ntype,           &
              natom,           &
              natomi,          & 
              ndynamic,        &
              ncoord,          &
              pripos,          &
              prifix)
            
! End of writing the POSCAR file

END SUBROUTINE recell
