!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
!                         Standardized Unit Cell
!
!        >>>>>>>>>>>>>>>>>>  Trianglar system  <<<<<<<<<<<<<<<<<<
!      
!        b axis is taken as the unique axis
!      
!        alpha = 90 degree and bata = 90 degree
!        gamma > 90 degree
! 
!        a > b
!       
!        a is set along +x direction of Cartesian coordinates
!        b is set in x-y plane of Cartesian coordinates      
!        c is set along +z direction of Cartesian coordinates
!        
!
!        >>>>>>>>>>>>>>>>>  Rectanglar system  <<<<<<<<<<<<<<<<<      
!
!      
!        alpha = beta = gamma = 90 degree
!
!        a > b
!      
!        a is set along +x direction of Cartesian coordinates
!        b is set along +y direction of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!
!
!        >>>>>>>>>>>>>>>>>>  Square system  <<<<<<<<<<<<<<<<<<      
!      
!
!        alpha = beta = gamma = 90 degree
!      
!        a = b
!      
!        a is set along +x direction of Cartesian coordinates
!        b is set along +y direction of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!
!      
!        >>>>>>>>>>>>>>>>>>   Hexagon system   <<<<<<<<<<<<<<<<<<      
!
!      
!        alpha = beta = 90 degree
!        gamma = 120 degree
!      
!        a = b
!      
!        a is set along +x direction of Cartesian coordinates
!        b is set in x-y plane of Cartesian coordinates
!        c is set along +z direction of Cartesian coordinates
!
!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

! MODULE defs_basis
! IMPLICIT NONE
! INTEGER, PARAMETER :: dp=KIND(1.00D0)
! END MODULE defs_basis

SUBROUTINE recell2d
!
!  Purpose:
!
!    To transform the structure to IEEE-format for 2D materials
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    space_group: The space group number
!    refine_cell:
!    lattice   : The vectors defining the unit cell of the system with lattice(i,j)=privect(i,j)
!    positions : Three coordinates for each atom with positions(j,i)=pripos(i,j)
!    pricoord  : Direct or Cartesian
!
!    lega(b,c) : The length of privect(i,*)
!    lega(b,c)r: The length of recipvect(i,*)
!    cos_alpha : The angle between privect(2,*) and privect(3,*)
!    cos_bata  : The angle between privect(3,*) and privect(1,*)
!    cos_gamma : The angle between privect(1,*) and privect(2,*)
!    cos_alphar: The angle between recipvect(2,*) and recipvect(3,*)
!    cos_batar : The angle between recipvect(3,*) and recipvect(1,*)
!    cos_gammar: The angle between recipvect(1,*) and recipvect(2,*)
!
!    recipvect : The reciprocal vector of redvect
!
!
!
!

USE defs_basis
USE spglib_f08
USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk, m, nn
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: space_group, refine_cell

DOUBLE PRECISION :: lattice(3,3), positions(3,ATOMMAX)
DOUBLE PRECISION :: lega, legb, legc
DOUBLE PRECISION :: legar, legbr, legcr
DOUBLE PRECISION :: cos_alpha, cos_bata, cos_gamma
DOUBLE PRECISION :: cos_alphar, cos_batar, cos_gammar
DOUBLE PRECISION :: cos_angle
DOUBLE PRECISION :: temp, temppos(ATOMMAX,3), tempvect(5,5), recipvect(3,3)

CHARACTER(len=20) :: pricoord, symbol

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! Output the initial structure

IF ( ncoord .EQ. 0 ) THEN
        pricoord='Direct'
ELSEIF ( ncoord .EQ. 1 ) THEN
        pricoord='Cartesian'
ENDIF

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*)
WRITE(19,*) ' RECELL STRUCTURE (2D)'
WRITE(19,*) ' ----------------------------------------------------------------------------- '

WRITE(19,*) '| The initial structure:'
WRITE(19,*) '| '

WRITE(19,'(A3,F8.5)') '| ', alat
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ', (ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ', (natomi(i), i=1,ntype)
WRITE(19,'(A3,A9)') '| ', pricoord
DO i=1, natom
   WRITE(19,"(A3,I4,3F20.10)") '| ', atom_type(i), (pripos(i,j),j=1,3)
ENDDO

CLOSE(19)      

! End of outputing the initial structure 

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

! To get the space group of the structure "INPOS" 

DO i=1,3
   DO j=1,3
      lattice(i,j)=privect(i,j)
   ENDDO
ENDDO

DO i=1,natom
   DO j=1,3
      positions(j,i)=pripos(i,j)
   ENDDO
ENDDO

symbol='             '
space_group=spg_get_international(symbol,lattice,positions,atom_type,natom,SYMPREC)

IF ( space_group .EQ. 0 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Cannot get space group!                                                  |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP
ENDIF

! To refine the new cell with standardized unit cell

refine_cell=spg_refine_cell(lattice,positions,atom_type,natom,SYMPREC)

IF ( refine_cell .EQ. 0 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Cannot recell the structure!                                             |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP     
ENDIF

DO i=1,3
   DO j=1,3
      privect(i,j)=lattice(i,j)
   ENDDO
ENDDO

!! refine_cell is the atom number in the recelled structure

natom=refine_cell

DO i=1, ntype
   natomi(i)=0
   DO j=1, natom
      IF ( atom_type(j) .EQ. i ) THEN
              natomi(i)=natomi(i)+1
      ENDIF
   ENDDO
ENDDO

m=1
DO i=1,ntype
   DO j=1,natom
      IF ( atom_type(j) .EQ. i ) THEN
              DO k=1,3
                 pripos(m,k)=positions(k,j)
              ENDDO
              m=m+1 
      ENDIF
   ENDDO
ENDDO

! Rectanglar lattice: b < a < c

IF ( space_group.GE.16 .AND. space_group.LE.74 ) THEN

        DO i=1,2                
           IF ( privect(3,3) .LT. privect(i,i) ) THEN
                   temp=privect(3,3)
                   privect(3,3)=privect(i,i)
                   privect(i,i)=temp

                   DO j=1,natom
                      temp=pripos(j,3)
                      pripos(j,3)=pripos(j,i)
                      pripos(j,i)=temp
                   ENDDO
           ENDIF
        ENDDO

        IF ( privect(1,1) .LT. privect(2,2) ) THEN
                temp=privect(1,1)
                privect(1,1)=privect(2,2)
                privect(2,2)=temp

                DO j=1,natom
                   temp=pripos(j,1)
                   pripos(j,1)=pripos(j,2)
                   pripos(j,2)=temp
                ENDDO
        ENDIF            

ENDIF

! Trianglar lattice: c <-> b

IF ( space_group.GE.3 .AND. space_group.LE.15 ) THEN
        
        temp=privect(2,2)
        privect(2,2)=privect(3,3)
        privect(3,3)=temp

        privect(2,1)=privect(3,1)
        privect(3,1)=0.0000
             
        DO j=1,natom
           temp=pripos(j,3)
           pripos(j,3)=pripos(j,2)
           pripos(j,2)=temp
        ENDDO

ENDIF

! Trianglar lattice: b < a

IF ( space_group.GE.3 .AND. space_group.LE.15 ) THEN

        lega=SQRT(privect(1,1)**2+privect(1,2)**2+privect(1,3)**2)
        legb=SQRT(privect(2,1)**2+privect(2,2)**2+privect(2,3)**2)

        IF ( lega .LT. legb ) THEN
               privect(1,1)=legb

               privect(2,2)=lega*privect(2,2)/legb
               privect(2,1)=lega*privect(2,1)/legb
               
               DO j=1,natom
                   temp=pripos(j,1)
                   pripos(j,1)=pripos(j,2)
                   pripos(j,2)=temp
               ENDDO
       ENDIF
        
ENDIF

! Create the POSCAR for first-principles calculation

!! output the recelled structure

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*) '| '
WRITE(19,*) '| The recelled structure:'
WRITE(19,*) '| '

WRITE(19,'(A3,F8.5)') '| ',alat
DO i=1,3
   WRITE(19,"(A3,3F20.10)") '| ', (privect(i,j),j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ', (ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ', (natomi(i), i=1,ntype)
WRITE(19,'(A3,A9)') '| ', pricoord
DO i=1, natom
   WRITE(19,"(A3,I4,3F20.10)") '| ', atom_type(i), (pripos(i,j),j=1,3)
ENDDO

CLOSE(19)

!! end of outputing the recelled structure   

! To write the POSCAR file

outpos_name='IEEEPOS'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              privect,         &
              ele_symbol,      & 
              ntype,           &
              natom,           &
              natomi,          & 
              ndynamic,        &
              ncoord,          &
              pripos,          &
              prifix)
            
! End of writing the POSCAR file

END SUBROUTINE recell2d
