SUBROUTINE redefinelat(new_lat_vect)
!
!  Purpose:
!
!    To redefine new right-handed lattice vectors "new_lat_vect" of the current structure
!    for 3D and 2D materials. The new lattice vectors a_new and b_new will be put to be
!    parallel to x-axis and to lie in x-y plane, respectively
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!    sup_range : The range of supercell
!    lower     :
!    upper     :
!    sup_range_tot: sup_range(1)*sup_range(2)*sup_range(3)
!    new_lat_new: The new lattice vector for redefining lattive vector (3x3) 

!    sup_natom : Total number of atoms for supercell
!    sup_natomi: Number of atoms per atomic species for supercell
!    sup_atom_type: Array for atomic type using in SPGLIB for supercell
!    supvect   : The vectors defining the unit cell of the system for supercell
!    suppos    : Three coordinates for each atom for supercell
!
!    red_natom : Total number of atoms for redefined lattice
!    red_natomi: Number of atoms per atomic species for redefined lattice
!    redvect   : The vectors defining the unit cell of the system for redefined lattice
!    redvectI  : The inverse matrix of redvect matrix
!    redpos    : Three coordinates for each atom for redefined lattice
!
!    lega(b,c) : The length of redvect(i,*)
!    lega(b,c)r: The length of recipvect(i,*)
!    cos_alpha : The angle between redvect(2,*) and redvect(3,*)
!    cos_bata  : The angle between redvect(3,*) and redvect(1,*)
!    cos_gamma : The angle between redvect(1,*) and redvect(2,*)
!    cos_alphar: The angle between recipvect(2,*) and recipvect(3,*)
!    cos_batar : The angle between recipvect(3,*) and recipvect(1,*)
!    cos_gammar: The angle between recipvect(1,*) and recipvect(2,*)
!
!    recipvect : The reciprocal vector of redvect
!    
!    supcell_rge: The range of supercell, with supvect(3,3) >15.00 Ang
!
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

! INTEGER :: sup_range(3), lower(3), upper(3)
! INTEGER :: sup_range_tot
! INTEGER :: new_lat_vect(3,3)

INTEGER :: countnum
INTEGER :: sup_natom
INTEGER :: sup_natomi(TYPEMAX), sup_atom_type(ATOMMAX)
INTEGER :: red_natom
INTEGER :: red_natomi(TYPEMAX)

DOUBLE PRECISION :: sup_range(3), lower(3), upper(3)
DOUBLE PRECISION :: sup_range_tot
DOUBLE PRECISION :: new_lat_vect(3,3)

DOUBLE PRECISION :: lega, legb, legc
DOUBLE PRECISION :: legar, legbr, legcr
DOUBLE PRECISION :: cos_alpha, cos_bata, cos_gamma
DOUBLE PRECISION :: cos_alphar, cos_batar, cos_gammar
DOUBLE PRECISION :: cos_angle

DOUBLE PRECISION :: recipvect(3,3), tmpvect(5,5)
DOUBLE PRECISION :: supvect(3,3), suppos(ATOMMAX,3), suppos_tmp(ATOMMAX,3)
DOUBLE PRECISION :: redvect(3,3), redvectI(3,3), redpos(ATOMMAX,3)

! The variables for creating supercell

INTEGER :: supcell_rge(3)

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

DO i=1,natom
   DO j=1,3
      DO
         IF ( (pripos(i,j).GT.0.0000 .OR. ABS(pripos(i,j)).LT.ZERO_TOLERANCE) .AND. pripos(i,j).LT.(1.0000-ZERO_TOLERANCE) ) THEN
                 EXIT
         ELSEIF ( pripos(i,j) .LT. ZERO_TOLERANCE ) THEN
                 pripos(i,j)=pripos(i,j)+1.00D0
         ELSEIF ( pripos(i,j) .GT. (1.0000-ZERO_TOLERANCE) ) THEN
                 pripos(i,j)=pripos(i,j)-1.00D0
         ENDIF
      ENDDO   
   ENDDO
ENDDO

!---------------------------------------------------

! To bulid the supercell

DO i=1, 3
   lower(i)=0
   upper(i)=0
   DO j=1, 3
      IF ( new_lat_vect(j,i)<0 ) THEN
              lower(i)=lower(i)+new_lat_vect(j,i)
      ELSE
              upper(i)=upper(i)+new_lat_vect(j,i)
      ENDIF
   ENDDO
ENDDO

! To 

DO i=1, 3
   sup_range(i)=upper(i)-lower(i)
ENDDO

sup_range_tot=1
DO i=1, 3
   sup_range_tot=sup_range_tot*sup_range(i)
ENDDO

sup_natom=natom*sup_range_tot

DO i=1, ntype
   sup_natomi(i)=natomi(i)*sup_range_tot
ENDDO

! To

DO ii=1, natom
   DO i=1, sup_range(1)
      DO j=1, sup_range(2)
         DO k=1, sup_range(3)
            countnum=(ii-1)*sup_range_tot+(i-1)*sup_range(2)*sup_range(3)+(j-1)*sup_range(3)+k
            suppos(countnum,1)=pripos(ii,1)+i-1+lower(1)
            suppos(countnum,2)=pripos(ii,2)+j-1+lower(2)
            suppos(countnum,3)=pripos(ii,3)+k-1+lower(3)
         ENDDO
      ENDDO
   ENDDO
ENDDO

!------------------------------------------------------------

k=0
DO i=1, ntype
   DO j=1, sup_natomi(i)
      k=k+1
      sup_atom_type(k)=i
   ENDDO
ENDDO

! 

DO i=1, 3
   DO j=1, 3
      redvect(i,j)=0
      DO k=1, 3
         redvect(i,j)=redvect(i,j)+new_lat_vect(i,k)*privect(k,j)
      ENDDO
   ENDDO
ENDDO

DO i=1, 3
   DO j=1, 3
      redvectI(i,j)=redvect(i,j)
   ENDDO
ENDDO
call brinv(redvectI,3)

!--------------------------------------------------

! 

DO i=1, sup_natom
   
   DO j=1, 3
      suppos_tmp(i,j)=0.0000
      DO k=1, 3
         suppos_tmp(i,j)=suppos_tmp(i,j)+suppos(i,k)*privect(k,j)
      ENDDO
   ENDDO
   DO j=1, 3
      suppos(i,j)=suppos_tmp(i,j)
   ENDDO

   DO j=1, 3
      suppos_tmp(i,j)=0.0000
      DO k=1, 3
         suppos_tmp(i,j)=suppos_tmp(i,j)+suppos(i,k)*redvectI(k,j)
      ENDDO
   ENDDO
   DO j=1, 3
      suppos(i,j)=suppos_tmp(i,j)
   ENDDO

ENDDO

red_natom=0
DO i=1, ntype
   red_natomi(i)=0
ENDDO

!------------------------------------------------------------

DO i=1, sup_natom

   IF ( (suppos(i,1).GT.0.0000 .OR. ABS(suppos(i,1)).LT.ZERO_TOLERANCE) .AND. suppos(i,1).LT.(1.0000-ZERO_TOLERANCE) .AND. &
        (suppos(i,2).GT.0.0000 .OR. ABS(suppos(i,2)).LT.ZERO_TOLERANCE) .AND. suppos(i,2).LT.(1.0000-ZERO_TOLERANCE) .AND. &
        (suppos(i,3).GT.0.0000 .OR. ABS(suppos(i,3)).LT.ZERO_TOLERANCE) .AND. suppos(i,3).LT.(1.0000-ZERO_TOLERANCE) ) THEN

           red_natom=red_natom+1
           DO j=1, 3
              redpos(red_natom,j)=suppos(i,j)
           ENDDO
           red_natomi(sup_atom_type(i))=red_natomi(sup_atom_type(i))+1
   ENDIF

ENDDO

! Project and Rotation

lega=SQRT(redvect(1,1)**2+redvect(1,2)**2+redvect(1,3)**2)
legb=SQRT(redvect(2,1)**2+redvect(2,2)**2+redvect(2,3)**2)
legc=SQRT(redvect(3,1)**2+redvect(3,2)**2+redvect(3,3)**2)

cos_alpha=(redvect(2,1)*redvect(3,1)+redvect(2,2)  &
          *redvect(3,2)+redvect(2,3)*redvect(3,3))/legb/legc
cos_bata=(redvect(1,1)*redvect(3,1)+redvect(1,2)   &
         *redvect(3,2)+redvect(1,3)*redvect(3,3))/lega/legc
cos_gamma=(redvect(1,1)*redvect(2,1)+redvect(1,2)  &
          *redvect(2,2)+redvect(1,3)*redvect(2,3))/lega/legb

! define the reciprocal matrix of privect

DO i=1,3
   DO j=1,3
      tmpvect(i,j)=redvect(i,j)
   ENDDO
ENDDO
DO i=1,3
   DO j=4,5
      tmpvect(i,j)=tmpvect(i,j-3)
   ENDDO
ENDDO
DO i=4,5
   DO j=1,5
      tmpvect(i,j)=tmpvect(i-3,j)
   ENDDO
ENDDO
                
DO i=1,3
   DO j=1,3
      recipvect(i,j)=tmpvect(i+1,j+1)*tmpvect(i+2,j+2)  &
                    -tmpvect(i+1,j+2)*tmpvect(i+2,j+1)
   ENDDO
ENDDO

! End of defining the reciprocal matrix of privect        

legar=SQRT(recipvect(1,1)**2+recipvect(1,2)**2+recipvect(1,3)**2)
legbr=SQRT(recipvect(2,1)**2+recipvect(2,2)**2+recipvect(2,3)**2)
legcr=SQRT(recipvect(3,1)**2+recipvect(3,2)**2+recipvect(3,3)**2)

cos_alphar=(redvect(1,1)*recipvect(1,1)+redvect(1,2)  &
           *recipvect(1,2)+redvect(1,3)*recipvect(1,3))/lega/legar
cos_batar=(redvect(2,1)*recipvect(2,1)+redvect(2,2)   &
          *recipvect(2,2)+redvect(2,3)*recipvect(2,3))/legb/legbr
cos_gammar=(redvect(3,1)*recipvect(3,1)+redvect(3,2)  &
           *recipvect(3,2)+redvect(3,3)*recipvect(3,3))/legc/legcr

! Coplanar

IF ( ABS(cos_alphar).LT.ZERO_TOLERANCE .OR.  &
     ABS(cos_batar).LT.ZERO_TOLERANCE .OR.   &  
     ABS(cos_gammar).LT.ZERO_TOLERANCE ) THEN

        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    The input new lattice vectors are coplanar as redefining the lattice!    |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP        

ENDIF

! Right-handed set of axis vectors

IF ( cos_alphar.LT.0.0000 .OR. &
     cos_batar .LT.0.0000 .OR.  &
     cos_gammar.LT.0.0000 ) THEN

        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    The input new lattice vectors do not form a right-handed set of axis     |'
        WRITE(19,*) '|    vectors as redefining the lattice!                                       |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP   

ENDIF

! To

! >>> a
redvect(1,1)=lega
redvect(1,2)=0.0000
redvect(1,3)=0.0000

! >>> b
IF ( ABS(cos_gamma).LT.ZERO_TOLERANCE ) THEN
        redvect(2,1)=0.0000
ELSE
        redvect(2,1)=legb*cos_gamma
ENDIF
redvect(2,2)=SQRT(ABS(legb**2-redvect(2,1)**2))
redvect(2,3)=0.0000

! >>> c
IF ( ABS(cos_bata).LT.ZERO_TOLERANCE ) THEN
        redvect(3,1)=0.0000
ELSE
        redvect(3,1)=legc*cos_bata
ENDIF

redvect(3,3)=legc*cos_gammar

IF ( ABS(cos_gammar-1.0000).LT.ZERO_TOLERANCE ) THEN
        redvect(3,2)=0.0000
ELSE
        redvect(3,2)=SQRT(ABS(legc**2-redvect(3,1)**2-redvect(3,3)**2))
ENDIF

!---------------------------------------------------

! To make the Angle unchanged

cos_angle=(redvect(2,1)*redvect(3,1)+  &
           redvect(2,2)*redvect(3,2)+  &
           redvect(2,3)*redvect(3,3))/legb/legc

IF ( ABS(cos_angle-cos_alpha) .GT. ZERO_TOLERANCE ) THEN
        redvect(3,2)=-redvect(3,2);
ENDIF

! To output the results

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
WRITE(19,*)
WRITE(19,*) ' REDEFINE LATTICE'
WRITE(19,*) ' ----------------------------------------------------------------------------- '
WRITE(19,*) '| Before redefining the lattice, the POS file:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ',(ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ',(natomi(i), i=1,ntype)
DO i=1, natom
   WRITE(19,'(A3,3F20.10)') '| ',(pripos(i,j), j=1,3)
ENDDO

WRITE(19,*) '| '
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| New lattice vector'
DO i=1,3
!   WRITE(19,'(A11,3I5)') '|         ', (new_lat_vect(i,j), j=1,3)
   WRITE(19,'(A11,3F8.2)') '|         ', (new_lat_vect(i,j), j=1,3)
ENDDO
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| '

WRITE(19,*) '| After redefining the lattice, the POS file:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (redvect(i,j), j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ',(ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ',(red_natomi(i), i=1,ntype)
DO i=1, red_natom
   WRITE(19,'(A3,3F20.10)') '| ',(redpos(i,j), j=1,3)
ENDDO

CLOSE(19)

! To write the POSCAR file

outpos_name='REDPOS'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              redvect,         &
              ele_symbol,      & 
              ntype,           &
              red_natom,       &
              red_natomi,      & 
              ndynamic,        &
              ncoord,          &
              redpos,          &
              prifix)

! To create supercell

supcell_rge(1)=1
supcell_rge(2)=1
supcell_rge(3)=CEILING(15.0000/redvect(3,3))

CALL supercell(outpos_name, supcell_rge)

! End of writing the POSCAR file

END SUBROUTINE redefinelat
