SUBROUTINE rotproj(rotangle)
!
!  Purpose:
!
!    Rotation method of projection
!
!    To project one specific orientation (i.e., the imposed strain direction) of a given crystal
!    to be parallel to one axis vector (e.g. y) for affine tensile deformation, and projects 
!    one axis vector (e.g. x) to be perpendicular to the slip plane and another one (e.g. y) to
!    be parallel to the slip direction in that plane for the affine shearing deformation
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    rotangle  : The rotation angle of x, y and z axes (Unit: deg)
!    rotangle_pi: rotangle_pi=rotangle/180 (Unit: pi)
!    rotx(y,z)mat: The projection matrix along x, y and z axes
!    rotxvect  : The vectors defining the unit cell of the system after rotating along x axis
!    rotyvect  : The vectors defining the unit cell of the system after rotating along x and y axes
!    rotzvect  : The vectors defining the unit cell of the system after rotating along x, y and z axes
!    rotvect   : The vectors defining the unit cell of the system after projection
!
!
!
!
!  Note:
!    To operate the lattice vectors by matrix
!    ---------------------------------------------
!         |a11 a12 a13|       |mat11 mat12 mat13|
!    R=   |a21 a22 a23|  MAT= |mat21 mat22 mat23|
!         |a31 a32 a33|       |mat31 mat32 mat33|
!    ---------------------------------------------
!              |aa11 aa12 aa13|
!    RR=R*MAT= |aa21 aa22 aa23|
!              |aa31 aa32 aa33|
!    ---------------------------------------------
!    The rotation angle of x, y and z axes: x, y, z
!
!        |1       0      0| |cos(y) 0 -sin(y)| | cos(z) sin(z) 0|
!    MAT=|0  cos(x) sin(x)|*|     0 1       0|*|-sin(z) cos(z) 0|
!        |0 -sin(x) cos(x)| |sin(y) 0  cos(y)| |      0      0 1|
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

DOUBLE PRECISION :: rotangle(3), rotangle_pi(3)
DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: rotxmat, rotymat, rotzmat
DOUBLE PRECISION :: rotvect(3,3)
DOUBLE PRECISION :: rotxvect(3,3), rotyvect(3,3), rotzvect(3,3)

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

! 

DO i=1,3
   rotangle(i)=rotangle(i)/180.0000*PI
ENDDO

! The rotation matrix rotxmat

ALLOCATE(rotxmat(3,3))
rotxmat(:,:)=RESHAPE((/1.00D0,  0.00D0,            0.00D0,           &
                       0.00D0,  COS(rotangle(1)), -SIN(rotangle(1)), &
                       0.00D0,  SIN(rotangle(1)),  COS(rotangle(1))/),(/3,3/))

! xx axis rotation
DO i=1,3
   DO j=1,3
      rotxvect(i,j)=0.0000
      DO k=1,3
         rotxvect(i,j)=rotxvect(i,j)+privect(i,k)*rotxmat(k,j)
      ENDDO
   ENDDO
ENDDO

DEALLOCATE(rotxmat)

! The rotation matrix rotymat

ALLOCATE(rotymat(3,3))
rotymat(:,:)=RESHAPE((/COS(rotangle(2)),  0.00D0,  SIN(rotangle(2)), &
                       0.00D0,            1.00D0,  0.00D0,           &
                      -SIN(rotangle(2)),  0.00D0,  COS(rotangle(2))/),(/3,3/))

! yy axis rotation
DO i=1,3
   DO j=1,3
      rotyvect(i,j)=0.0000
      DO k=1,3
         rotyvect(i,j)=rotyvect(i,j)+rotxvect(i,k)*rotymat(k,j)
      ENDDO
   ENDDO
ENDDO

DEALLOCATE(rotymat)

! The rotation matrix rotzmat

ALLOCATE(rotzmat(3,3))
rotzmat(:,:)=RESHAPE((/COS(rotangle(3)), -SIN(rotangle(3)),  0.00D0, &
                       SIN(rotangle(3)),  COS(rotangle(3)),  0.00D0, &
                       0.00D0,            0.00D0,            1.00D0/),(/3,3/))

! zz axis rotation
DO i=1,3
   DO j=1,3
      rotzvect(i,j)=0.0000
      DO k=1,3
         rotzvect(i,j)=rotzvect(i,j)+rotyvect(i,k)*rotzmat(k,j)
      ENDDO
   ENDDO
ENDDO

DEALLOCATE(rotzmat)

DO i=1,3
   DO j=1,3
      rotvect(i,j)=rotzvect(i,j)
   ENDDO
ENDDO

! To output the results

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
WRITE(19,*)
WRITE(19,*) ' CRYSTAL PROJECTION (ROTPROJ)'
WRITE(19,*) ' ----------------------------------------------------------------------------- '
WRITE(19,*) '| Before Projction, the three lattice vectors:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO

DO i=1,3
   rotangle_pi(i)=rotangle(i)/PI
ENDDO

WRITE(19,*) '| '
WRITE(19,*) '| -----------------------------------------'
WRITE(19,'(A,3(F6.3,A2))') ' | The rotation angle of x, y and z axes:', (rotangle_pi(j),'pi',j=1,3)
WRITE(19,*) '| -----------------------------------------'
WRITE(19,*) '| '

WRITE(19,*) '| After Projction, the three lattice vectors:' 
WRITE(19,*) '| '
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (rotvect(i,j), j=1,3)
ENDDO


CLOSE(19)

! To write the POSCAR file

outpos_name='PROJPOS'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              rotvect,         &
              ele_symbol,      & 
              ntype,           &
              natom,           &
              natomi,          & 
              ndynamic,        &
              ncoord,          &
              pripos,          &
              prifix)
            
! End of writing the POSCAR file

END SUBROUTINE rotproj
