SUBROUTINE strdisp
!
!  Purpose:
!
!    To determine the variation of atom coordination with direct mode as a function
!    of strain value under affine pure or simple deformation for 3D materials, or 
!    uniaxial or biaxial tensile deformation for 2D materials
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    istrvalue : The accumulated strain value
!    strain    : The matrix of istrvalue value, if AFFPOS_[istrvalue].vasp
!    ten_str   : The weight for tensile strain
!    shear_str : The weight for shearing strain
!    istr0     : The initial strain value
!    step_length: The step length for affine deformation
!    step_num  : The step number for affine deformation
!
!    strpos    : Three coordinates for each atom under strain with Direct mode
!    strpos1   : POS(STR(i))-POS(STR(0))
!    strpos2   : POS(STR(i))-POS(STR(i-1))
!
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: strain_num
DOUBLE PRECISION :: istrvalue, strain(STRAINMAX)
DOUBLE PRECISION :: ten_str, shear_str, istr0, step_length, step_num
DOUBLE PRECISION :: strpos(STRAINMAX,ATOMMAX,3)
DOUBLE PRECISION :: strpos1(STRAINMAX,ATOMMAX,3)
DOUBLE PRECISION :: strpos2(STRAINMAX,ATOMMAX,3)

CHARACTER(len=30) :: filename
CHARACTER(len=20) :: titstr1, titstr2
CHARACTER(len=20) :: tmpstr

LOGICAL :: filefound

! To read the istrvalue values from LOOPCAR

OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
READ(87,*) titstr1, titstr2

IF ( TRIM(titstr1) .EQ. 'AFFINE' ) THEN
        READ(87,*)
        READ(87,*) tmpstr, tmpstr, tmpstr, ten_str
        READ(87,*) tmpstr, tmpstr, tmpstr, shear_str
        READ(87,*) tmpstr, tmpstr, tmpstr, istr0
        READ(87,*) tmpstr, tmpstr, tmpstr, step_length
        READ(87,*) tmpstr, tmpstr, tmpstr, step_num
        CLOSE(87)
ELSE
        CLOSE(87)
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    "fconv" mode can be obtained only for affine deformation!                |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP    
ENDIF

! To read the POSFILE AFFPOS_[istrvalue].vasp

strain_num=0

DO i=0,INT(step_num)

   istrvalue=istr0+step_length*i
   WRITE(filename,'(A7,F6.4,A5)') 'AFFPOS_', istrvalue, '.vasp'

   INQUIRE (FILE=filename, EXIST=filefound)
   IF ( filefound ) THEN

           strain_num=strain_num+1
           strain(strain_num)=istrvalue
           ! To read the POSCAR file
           CALL readpos(filename,        &
                        pos_title,       &
                        alat,            &
                        privect,         &
                        ele_symbol,      &
                        ntype,           &
                        natom,           &
                        natomi,          &
                        ndynamic,        &
                        ncoord,          &
                        pripos,          &
                        prifix,          &
                        atom_type)

   ENDIF

   DO j=1,natom
      DO k=1,3

         IF ( strain_num .GE. 2 ) THEN
                 IF ( ABS(pripos(j,k)-strpos(strain_num-1,j,k)) .GT. ABS(pripos(j,k)+1.0000-strpos(strain_num-1,j,k)) ) THEN
                         pripos(j,k)=pripos(j,k)+1.0000
                 ELSEIF ( ABS(pripos(j,k)-strpos(strain_num-1,j,k)) .GT. ABS(pripos(j,k)-1.0000-strpos(strain_num-1,j,k)) ) THEN
                         pripos(j,k)=pripos(j,k)-1.0000
                 ENDIF
         ENDIF

         strpos(strain_num,j,k)=pripos(j,k)
         IF ( strain_num .GE. 2 ) THEN
                 strpos1(strain_num,j,k)=pripos(j,k)-strpos(1,j,k)
                 strpos2(strain_num,j,k)=pripos(j,k)-strpos(strain_num-1,j,k)
         ENDIF
      ENDDO
   ENDDO

   CALL writepos(filename,        &
                 pos_title,       &
                 alat,            &
                 privect,         &
                 ele_symbol,      &
                 ntype,           &
                 natom,           &
                 natomi,          &
                 ndynamic,        &
                 ncoord,          &
                 pripos,          &
                 prifix)

ENDDO

! To write the file RDISP

! OPEN(UNIT=67, FILE='RDISP', STATUS='unknown')
! DO i=1,natom
!    WRITE(67,'(A5,I03.3,A)') 'ATOM(', i, ')'
!    WRITE(67,*)
! 
!    WRITE(67,'(A108)') '   Strain               POS(STR(I))               POS(STR(i))-POS(STR(0))          POS(STR(i))-POS(STR(i-1))'
!    WRITE(67,'(A108)') '                  ax        by        cz           ax        by        cz          ax        by         cz  '
! 
!    WRITE(67,'(F10.6,A3,3F10.6)') strain(1), '  |', (strpos(1,i,k),k=1,3)
!    DO j=2,strain_num
!       WRITE(67,'(F10.6,3(A3,3F10.6))') strain(j), '  |', (strpos(j,i,k),k=1,3), '  |', (strpos1(j,i,k),k=1,3), '  |', (strpos2(j,i,k),k=1,3)
!    ENDDO
!    WRITE(67,*)
! ENDDO

! CLOSE(67)

END SUBROUTINE strdisp
