MODULE defs_basis
IMPLICIT NONE
INTEGER, PARAMETER :: dp=KIND(1.00D0)
END MODULE defs_basis

SUBROUTINE symmetry(sympreci)
!
!  Purpose:
!
!    To determine the symmetry of structure and the possible tensile crystal direction and shearing
!    slip system based on the symmetry
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    sympreci  : The input tolerance of distance between atomic positions and the lengths of lattice vectors to be tolerated in the symmetry finding
!    spgnum    : The determined space group number 
!    space_groupi(s): The determined space group number 
!    lattice   : The vectors defining the unit cell of the system with lattice(i,j)=privect(i,j)
!    positions : Three coordinates for each atom with positions(j,i)=pripos(i,j)
!    primntaom : Total number of atoms in the primitive cell
!    priintaomi: Number of atoms per atomic species in the primitive cell
!    primvect  : The vectors defining the unit cell of the system in the primitive cell
!    primpos   : Three coordinates for each atom in the primitive cell
!
!
!
!

USE defs_basis
USE spglib_f08
USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk, m
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: space_groupi, spgnum
INTEGER :: space_groups
INTEGER :: find_primitive

INTEGER :: primnatom, primnatomi(TYPEMAX)
DOUBLE PRECISION :: primvect(3,3), primpos(ATOMMAX,3)

DOUBLE PRECISION :: sympreci
DOUBLE PRECISION :: lattice(3,3), positions(3,ATOMMAX)
CHARACTER(len=20) :: pricoord
CHARACTER(len=30) :: symboli, symbols
type(SpglibDataset) :: get_dataset

! To read the POSCAR file

inpos_name='INPOS'

CALL readpos(inpos_name,      &
             pos_title,       &
             alat,            &
             privect,         &
             ele_symbol,      & 
             ntype,           &
             natom,           &
             natomi,          & 
             ndynamic,        &
             ncoord,          &
             pripos,          &
             prifix,          &
             atom_type)

! End of reading the POSCAR file

! Output the initial structure

IF ( ncoord .EQ. 0 ) THEN
        pricoord='Direct'
ELSEIF ( ncoord .EQ. 1 ) THEN
        pricoord='Cartesian'
ENDIF

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*)
WRITE(19,*) ' PRIMITIVE CELL'
WRITE(19,*) ' ----------------------------------------------------------------------------- '

WRITE(19,*) '| The initial structure:'
WRITE(19,*) '| '

WRITE(19,'(A3,F8.5)') '| ', alat
DO i=1,3
   WRITE(19,'(A3,3F20.12)') '| ', (privect(i,j), j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ', (ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ', (natomi(i), i=1,ntype)
WRITE(19,'(A3,A9)') '| ', pricoord
DO i=1, natom
   WRITE(19,"(A3,I4,3F20.10)") '| ', atom_type(i), (pripos(i,j),j=1,3)
ENDDO

CLOSE(19)      

! End of outputing the initial structure 

! To direct
CALL directpos(alat,     &
               privect,  &
               natom,    &
               ncoord,   &
               pripos)

! To get the space group of the structure "INPOS" 

DO i=1,3
   DO j=1,3
      lattice(i,j)=privect(i,j)
   ENDDO
ENDDO

DO i=1,natom
   DO j=1,3
      positions(j,i)=pripos(i,j)
   ENDDO
ENDDO

IF ( ABS(sympreci+1) .LT. ZERO_TOLERANCE ) THEN
        sympreci=SYMPREC
ENDIF

symboli='             '
space_groupi=spg_get_international(symboli,lattice,positions,atom_type,natom,sympreci)
IF ( space_groupi .EQ. 0 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Cannot get space group!                                                  |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP
ELSE
        spgnum=space_groupi
ENDIF

symbols='             '
space_groups=spg_get_schoenflies(symbols,lattice,positions,atom_type,natom,sympreci)
! IF ( space_groups .EQ. 0 ) THEN
!         OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
!         WRITE(19,*)        
!         WRITE(19,*) ' ----------------------------------------------------------------------------- '
!         WRITE(19,*) '|                                     ERROR                                   |'
!         WRITE(19,*) '|                            -----------------------                          |'
!         WRITE(19,*) '|    Cannot get space group!                                                  |'
!         WRITE(19,*) ' ----------------------------------------------------------------------------- '
!         CLOSE(19)
!         STOP
! ENDIF

! get_dataset=spg_get_dataset(lattice,positions,atom_type,natom,sympreci)
! IF ( get_dataset%spacegroup_number .EQ. 0 ) THEN
!         OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
!         WRITE(19,*)        
!         WRITE(19,*) ' ----------------------------------------------------------------------------- '
!         WRITE(19,*) '|                                     ERROR                                   |'
!         WRITE(19,*) '|                            -----------------------                          |'
!         WRITE(19,*) '|    Cannot get the srystallographic database and the space group type!       |'
!         WRITE(19,*) ' ----------------------------------------------------------------------------- '
!         CLOSE(19)
!         STOP
! ENDIF

find_primitive=spg_find_primitive(lattice,positions,atom_type,natom,sympreci)
IF ( find_primitive .EQ. 0 ) THEN
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)        
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Cannot fine the primitive cell!                                          |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP
ENDIF

DO i=1,3
   DO j=1,3
      primvect(i,j)=lattice(i,j)
   ENDDO
ENDDO

!! find_primitive is the atom number in the primitive cell

primnatom=find_primitive

DO i=1, ntype
   primnatomi(i)=0
   DO j=1, primnatom
      IF ( atom_type(j) .EQ. i ) THEN
              primnatomi(i)=primnatomi(i)+1
      ENDIF
   ENDDO
ENDDO

m=1
DO i=1,ntype
   DO j=1,primnatom
      IF ( atom_type(j) .EQ. i ) THEN
              DO k=1,3
                 primpos(m,k)=positions(k,j)
              ENDDO
              m=m+1 
      ENDIF
   ENDDO
ENDDO

!! output the primitive cell

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*) '| '
WRITE(19,*) '| The primitive cell:'
WRITE(19,*) '| '

WRITE(19,'(A3,F8.5)') '| ',alat
DO i=1,3
   WRITE(19,"(A3,3F20.10)") '| ', (primvect(i,j),j=1,3)
ENDDO
WRITE(19,'(A3,10A5)') '| ', (ele_symbol(i), i=1,ntype)
WRITE(19,'(A3,10I4)') '| ', (primnatomi(i), i=1,ntype)
WRITE(19,'(A3,A9)') '| ', pricoord
DO i=1, primnatom
   WRITE(19,"(A3,I4,3F20.10)") '| ', atom_type(i), (primpos(i,j),j=1,3)
ENDDO

CLOSE(19)

!! end of outputing the primitive cell

! To write the POSCAR file

outpos_name='PRIMPOS'

CALL writepos(outpos_name,     &
              pos_title,       &
              alat,            &
              primvect,        &
              ele_symbol,      & 
              ntype,           &
              primnatom,       &
              primnatomi,      & 
              ndynamic,        &
              ncoord,          &
              primpos,         &
              prifix)
            
! End of writing the POSCAR file

! To write the HTCAR file for High-Throughout calculation

OPEN(UNIT=97, FILE='SYMMCAR', STATUS='unknown')

! Triclinic system
! Point group classes: 1, -1
!---------------------------------------------------------------------
!                           Triclinic system
! TENSILE CRYSTAL DIRECTION: [100], [010], [001], [110], [101], [011]
! SHEARING SLIP SYSTEM: (001)[100], (001)[010], (001)[110]
!                       (010)[100], (010)[001], (010)[101]
!                       (100)[010], (100)[001], (100)[011]
!                       (011)[100], (101)[010], (110)[001]
!                       (011)[01-1], (101)[10-1], (110)[1-10]
!----------------------------------------------------------------------
IF ( spgnum.GE.1 .AND. spgnum.LE.2 ) THEN
        
        WRITE(97,*) 'Triclinic system'
        WRITE(97,'(I4,A8,A8)') spgnum, TRIM(symboli), TRIM(symbols)
        WRITE(97,*) ' ---------------------------------------------------------- '
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '|   1  0  0  ![100]'
        WRITE(97,*) '|   0  1  0  ![010]'
        WRITE(97,*) '|   0  0  1  ![001]'
        WRITE(97,*) '|   1  1  0  ![110]'
        WRITE(97,*) '|   1  0  1  ![101]'
        WRITE(97,*) '|   0  1  1  ![011]'
        WRITE(97,*) '| ! SHEARING SLIP SYSTEM'
        WRITE(97,*) '|   0  0  1  1  0  0  !(001)[100]'
        WRITE(97,*) '|   0  0  1  0  1  0  !(001)[010]'
        WRITE(97,*) '|   0  0  1  1  1  0  !(001)[110]'
        WRITE(97,*) '|   0  1  0  1  0  0  !(010)[100]'
        WRITE(97,*) '|   0  1  0  0  0  1  !(010)[001]'
        WRITE(97,*) '|   0  1  0  1  0  1  !(010)[101]'
        WRITE(97,*) '|   1  0  0  0  1  0  !(100)[010]'
        WRITE(97,*) '|   1  0  0  0  0  1  !(100)[001]'
        WRITE(97,*) '|   1  0  0  0  1  1  !(100)[011]'
        WRITE(97,*) '|   0  1  1  1  0  0  !(011)[100]'
        WRITE(97,*) '|   1  0  1  0  1  0  !(101)[010]'
        WRITE(97,*) '|   1  1  0  0  0  1  !(110)[001]'
        WRITE(97,*) '|   0  1  1  0  1 -1  !(011)[01-1]'
        WRITE(97,*) '|   1  0  1  1  0 -1  !(101)[10-1]'
        WRITE(97,*) '|   1  1  0  1 -1  0  !(110)[1-10]'

! Monoclinic system
! Point group classes: m, 2, 2/m
!---------------------------------------------------------------------
!                           Monoclinic system
! TENSILE CRYSTAL DIRECTION: [100], [010], [001], [110], [101], [011]
! SHEARING SLIP SYSTEM: (001)[100], (001)[010], (001)[110]
!                       (010)[100], (010)[001], (010)[101]
!                       (100)[010], (100)[001], (100)[011]
!                       (011)[100], (101)[010], (110)[001]
!                       (011)[01-1], (101)[10-1], (110)[1-10]
!----------------------------------------------------------------------
ELSEIF ( spgnum.GE.3 .AND. spgnum.LE.15 ) THEN

        WRITE(97,*) 'Monoclinic system'
        WRITE(97,'(I4,A8,A8)') spgnum, TRIM(symboli), TRIM(symbols)
        WRITE(97,*) ' ---------------------------------------------------------- '
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '|   1  0  0  ![100]'
        WRITE(97,*) '|   0  1  0  ![010]'
        WRITE(97,*) '|   0  0  1  ![001]'
        WRITE(97,*) '|   1  1  0  ![110]'
        WRITE(97,*) '|   1  0  1  ![101]'
        WRITE(97,*) '|   0  1  1  ![011]'
        WRITE(97,*) '| ! SHEARING SLIP SYSTEM'
        WRITE(97,*) '|   0  0  1  1  0  0  !(001)[100]'
        WRITE(97,*) '|   0  0  1  0  1  0  !(001)[010]'
        WRITE(97,*) '|   0  0  1  1  1  0  !(001)[110]'
        WRITE(97,*) '|   0  1  0  1  0  0  !(010)[100]'
        WRITE(97,*) '|   0  1  0  0  0  1  !(010)[001]'
        WRITE(97,*) '|   0  1  0  1  0  1  !(010)[101]'
        WRITE(97,*) '|   1  0  0  0  1  0  !(100)[010]'
        WRITE(97,*) '|   1  0  0  0  0  1  !(100)[001]'
        WRITE(97,*) '|   1  0  0  0  1  1  !(100)[011]'
        WRITE(97,*) '|   0  1  1  1  0  0  !(011)[100]'
        WRITE(97,*) '|   1  0  1  0  1  0  !(101)[010]'
        WRITE(97,*) '|   1  1  0  0  0  1  !(110)[001]'
        WRITE(97,*) '|   0  1  1  0  1 -1  !(011)[01-1]'
        WRITE(97,*) '|   1  0  1  1  0 -1  !(101)[10-1]'
        WRITE(97,*) '|   1  1  0  1 -1  0  !(110)[1-10]'

! Orthorhombic system
! Point group classes: 222, mm2, 2/m2/m2/m
!---------------------------------------------------------------------
!                          Orthorhombic system
! TENSILE CRYSTAL DIRECTION: [100], [010], [001], [110], [101], [011]
! SHEARING SLIP SYSTEM: (001)[100], (001)[010], (001)[110]
!                       (010)[100], (010)[001], (010)[101]
!                       (100)[010], (100)[001], (100)[011]
!                       (011)[100], (101)[010], (110)[001]
!                       (011)[01-1], (101)[10-1], (110)[1-10]
!----------------------------------------------------------------------
ELSEIF ( spgnum.GE.16 .AND. spgnum.LE.74 ) THEN    

        WRITE(97,*) 'Orthorhombic system'
        WRITE(97,'(I4,A8,A8)') spgnum, TRIM(symboli), TRIM(symbols)
        WRITE(97,*) ' ---------------------------------------------------------- '
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '|   1  0  0  ![100]'
        WRITE(97,*) '|   0  1  0  ![010]'
        WRITE(97,*) '|   0  0  1  ![001]'
        WRITE(97,*) '|   1  1  0  ![110]'
        WRITE(97,*) '|   1  0  1  ![101]'
        WRITE(97,*) '|   0  1  1  ![011]'
        WRITE(97,*) '| ! SHEARING SLIP SYSTEM'
        WRITE(97,*) '|   0  0  1  1  0  0  !(001)[100]'
        WRITE(97,*) '|   0  0  1  0  1  0  !(001)[010]'
        WRITE(97,*) '|   0  0  1  1  1  0  !(001)[110]'
        WRITE(97,*) '|   0  1  0  1  0  0  !(010)[100]'
        WRITE(97,*) '|   0  1  0  0  0  1  !(010)[001]'
        WRITE(97,*) '|   0  1  0  1  0  1  !(010)[101]'
        WRITE(97,*) '|   1  0  0  0  1  0  !(100)[010]'
        WRITE(97,*) '|   1  0  0  0  0  1  !(100)[001]'
        WRITE(97,*) '|   1  0  0  0  1  1  !(100)[011]'
        WRITE(97,*) '|   0  1  1  1  0  0  !(011)[100]'
        WRITE(97,*) '|   1  0  1  0  1  0  !(101)[010]'
        WRITE(97,*) '|   1  1  0  0  0  1  !(110)[001]'
        WRITE(97,*) '|   0  1  1  0  1 -1  !(011)[01-1]'
        WRITE(97,*) '|   1  0  1  1  0 -1  !(101)[10-1]'
        WRITE(97,*) '|   1  1  0  1 -1  0  !(110)[1-10]'

! Tetragonal system
! Point group classes: 4, -4, 4/m, 422, 4mm, -42m, 4/m2/m/2/m
!---------------------------------------------------------------------
!                           Tetragonal system
! TENSILE CRYSTAL DIRECTION: [100], [001], [110], [101]
! SHEARING SLIP SYSTEM: (001)[100], (001)[110], (100)[010],
!                       (100)[001], (100)[011], (101)[010],
!                       (110)[001], (101)[10-1], (110)[1-10]
!----------------------------------------------------------------------
ELSEIF ( spgnum.GE.75 .AND. spgnum.LE.142 ) THEN

        WRITE(97,*) 'Tetragonal system'
        WRITE(97,'(I4,A8,A8)') spgnum, TRIM(symboli), TRIM(symbols)
        WRITE(97,*) ' ---------------------------------------------------------- '
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '|   1  0  0  ![100]'
        WRITE(97,*) '|   0  0  1  ![001]'
        WRITE(97,*) '|   1  1  0  ![110]'
        WRITE(97,*) '|   1  0  1  ![101]'
        WRITE(97,*) '| ! SHEARING SLIP SYSTEM'
        WRITE(97,*) '|   0  0  1  1  0  0  !(001)[100]'
        WRITE(97,*) '|   0  0  1  1  1  0  !(001)[110]'
        WRITE(97,*) '|   1  0  0  0  1  0  !(100)[010]'
        WRITE(97,*) '|   1  0  0  0  0  1  !(100)[001]'
        WRITE(97,*) '|   1  0  0  0  1  1  !(100)[011]'
        WRITE(97,*) '|   1  0  1  0  1  0  !(101)[010]'
        WRITE(97,*) '|   1  1  0  0  0  1  !(110)[001]'
        WRITE(97,*) '|   1  0  1  1  0 -1  !(101)[10-1]'
        WRITE(97,*) '|   1  1  0  1 -1  0  !(110)[1-10]'

! Trigonal system
! Point group classes: 3, -3, 32, 3m, -32/m
!---------------------------------------------------------------------
!                           Trigonal system
! TENSILE CRYSTAL DIRECTION: [0001], [11-20]
! SHEARING SLIP SYSTEM: (0001)[11-20], (0001)[10-10], (0001)[-1010]
!                       (10-10)[11-20], (10-11)[11-20], (10-12)[10-11]
!                       (11-22)[11-23]
!----------------------------------------------------------------------
ELSEIF ( spgnum.GE.143 .AND. spgnum.LE.167 ) THEN

        WRITE(97,*) 'Trigonal system'
        WRITE(97,'(I4,A8,A8)') spgnum, TRIM(symboli), TRIM(symbols)
        WRITE(97,*) ' ---------------------------------------------------------- '
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '|   0  0  1  ![0001]'  ! [0001]
        WRITE(97,*) '|   1  1  0  ![11-20]'  ! [11-20]
        WRITE(97,*) '| ! SHEARING SLIP SYSTEM'
        WRITE(97,*) '|   0  0  1  1  1  0  !(0001)[11-20]'  ! (0001)[11-20]
        WRITE(97,*) '|   0  0  1  2  1  0  !(0001)[10-10]'  ! (0001)[10-10]
        WRITE(97,*) '|   0  0  1 -2 -1  0  !(0001)[-1010]'  ! (0001)[-1010]
        WRITE(97,*) '|   1  0  0  0  1  0  !(10-10)[11-20]'  ! (10-10)[11-20]
        WRITE(97,*) '|   1  0  1  0  1  0  !(10-11)[11-20]'  ! (10-11)[11-20]
        WRITE(97,*) '|   1  0  2  2  1 -1  !(10-12)[10-11]'  ! (10-12)[10-11]
        WRITE(97,*) '|   1  1  2  1  1 -1  !(11-22)[11-23]'  ! (11-22)[11-23]

! Hexagonal system
! Point group classes: 6, -6, 6/m, 622, 6mm, -62m, 6/m2/m2/m
!---------------------------------------------------------------------
!                           Hexagonal system
! TENSILE CRYSTAL DIRECTION: [0001], [11-20]
! SHEARING SLIP SYSTEM: (0001)[11-20], (0001)[10-10], (0001)[-1010]
!                       (10-10)[11-20], (10-11)[11-20], (10-12)[10-11]
!                       (11-22)[11-23]
!----------------------------------------------------------------------
ELSEIF ( spgnum.GE.168 .AND. spgnum.LE.194 ) THEN

        WRITE(97,*) 'Hexagonal system'
        WRITE(97,'(I4,A8,A8)') spgnum, TRIM(symboli), TRIM(symbols)
        WRITE(97,*) ' ---------------------------------------------------------- '
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '|   0  0  1  ![0001]'  ! [0001]
        WRITE(97,*) '|   1  1  0  ![11-20]'  ! [11-20]
        WRITE(97,*) '| ! SHEARING SLIP SYSTEM'
        WRITE(97,*) '|   0  0  1  1  1  0  !(0001)[11-20]'  ! (0001)[11-20]
        WRITE(97,*) '|   0  0  1  2  1  0  !(0001)[10-10]'  ! (0001)[10-10]
        WRITE(97,*) '|   0  0  1 -2 -1  0  !(0001)[-1010]'  ! (0001)[-1010]
        WRITE(97,*) '|   1  0  0  0  1  0  !(10-10)[11-20]'  ! (10-10)[11-20]
        WRITE(97,*) '|   1  0  1  0  1  0  !(10-11)[11-20]'  ! (10-11)[11-20]
        WRITE(97,*) '|   1  0  2  2  1 -1  !(10-12)[10-11]'  ! (10-12)[10-11]
        WRITE(97,*) '|   1  1  2  1  1 -1  !(11-22)[11-23]'  ! (11-22)[11-23]

! Cubic system
! Point group classes: 23, 2/m-3, 432, -43m, 4/m-32/m
!---------------------------------------------------------------------
!                              Cubic system
! TENSILE CRYSTAL DIRECTION: [100], [110], [111]
! SHEARING SLIP SYSTEM: (001)[100], (110)[001], (110)[1-10],  
!                       (111)[1-10], (111)[11-2], (111)[-1-12]
!----------------------------------------------------------------------
ELSEIF ( spgnum.GE.195 .AND. spgnum.LE.230 ) THEN

        WRITE(97,*) 'Cubic system'
        WRITE(97,'(I4,A8,A8)') spgnum, TRIM(symboli), TRIM(symbols)
        WRITE(97,*) ' ---------------------------------------------------------- '
        WRITE(97,*) '| ! TENSILE CRYSTAL DIRECTION'
        WRITE(97,*) '|   1  0  0  ![100]'
        WRITE(97,*) '|   1  1  0  ![110]'
        WRITE(97,*) '|   1  1  1  ![111]'
        WRITE(97,*) '| ! SHEARING SLIP SYSTEM'
        WRITE(97,*) '|   0  0  1  1  0  0  !(001)[100]'
        ! WRITE(97,*) '|   0  0  1  1  1  0  !(001)[110]'
        WRITE(97,*) '|   1  1  0  0  0  1  !(110)[001]'
        WRITE(97,*) '|   1  1  0  1 -1  0  !(110)[1-10]'
        WRITE(97,*) '|   1  1  1  1 -1  0  !(111)[1-10]'
        WRITE(97,*) '|   1  1  1  1  1 -2  !(111)[11-2]'
        WRITE(97,*) '|   1  1  1 -1 -1  2  !(111)[-1-12]'

ENDIF

CLOSE(97)

END SUBROUTINE symmetry
