SUBROUTINE writebond(bondcut)
!
!  Purpose:
!
!    To analyze the variation of bond length as a function of the strain value under affine pure or 
!    simple deformation for 3D materials, or uniaxial or biaxial tensile deformation for 2D materials
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    bondcut   : The maximum value for Bond Length 
!    bond_num_atom: The bond number of per atom
!    dist_atom_indx: The atomic index, for which the atom in crystal forms bond with per atom
!    atom_atom_indx: The atomic index in the dist_indx matrix, for which the atom in 3x3x3 supercell forms bond with per atom
!    dist_direct: The coordinate difference (Direct mode) between per atom and the atoms in 3x3x3 supercell
!    dist_carte: The coordinate difference (Cartesian mode) between per atom and the atoms in 3x3x3 supercell
!    str_bond_lgth_atom: The bond length as a function of strain
!    dist_indx : The index matrix for 3x3x3 supercell
!
!    istrvalue : The accumulated strain value
!    strain    : The matrix of istrvalue value, if AFFPOS_[istrvalue].vasp
!    ten_str   : The weight for tensile strain
!    shear_str : The weight for shearing strain
!    istr0     : The initial strain value
!    step_length: The step length for affine deformation
!    step_num  : The step number for affine deformation
!
!
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: strain_num, bond_num_atom(ATOMMAX)
INTEGER :: atom_atom_indx(ATOMMAX,ATOMMAX), dist_atom_indx(ATOMMAX,ATOMMAX)
DOUBLE PRECISION :: bondcut
DOUBLE PRECISION :: istrvalue, strain(STRAINMAX)
DOUBLE PRECISION :: ten_str, shear_str, istr0, step_length, step_num
DOUBLE PRECISION :: dist_direct(ATOMMAX,ATOMMAX,3)
DOUBLE PRECISION :: dist_carte(ATOMMAX,ATOMMAX,3)
DOUBLE PRECISION ::str_bond_lgth_atom(ATOMMAX,ATOMMAX,STRAINMAX)
DOUBLE PRECISION,DIMENSION(:,:),ALLOCATABLE :: dist_indx

CHARACTER(len=30) :: filename
CHARACTER(len=20) :: strtmp 
CHARACTER(len=20) :: titstr1, titstr2
CHARACTER(len=20) :: tmpstr
CHARACTER(len=20) :: dist_atom_indx_tmp(ATOMMAX,ATOMMAX)

LOGICAL :: filefound

! To read the istrvalue values from LOOPCAR

OPEN(UNIT=87, FILE='LOOPCAR', STATUS='unknown')
READ(87,*) titstr1, titstr2

IF ( TRIM(titstr1) .EQ. 'AFFINE' ) THEN
        READ(87,*)
        READ(87,*) tmpstr, tmpstr, tmpstr, ten_str
        READ(87,*) tmpstr, tmpstr, tmpstr, shear_str
        READ(87,*) tmpstr, tmpstr, tmpstr, istr0
        READ(87,*) tmpstr, tmpstr, tmpstr, step_length
        READ(87,*) tmpstr, tmpstr, tmpstr, step_num
        CLOSE(87)
ELSE
        CLOSE(87)
        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    "bond" mode can be obtained only for affine deformation!                 |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP    
ENDIF

ALLOCATE(dist_indx(27,3))
dist_indx(:,:)=RESHAPE((/-1,-1,-1,-1,-1,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,  &
                         -1,-1,-1,0,0,0,1,1,1,-1,-1,-1,0,0,0,1,1,1,-1,-1,-1,0,0,0,1,1,1,  &
                         -1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1/),(/27,3/))

! To read the POSFILE AFFPOS_[istrvalue].vasp

WRITE(filename,'(A7,F6.4,A5)') 'AFFPOS_', istr0, '.vasp'
CALL bond_len(filename, bondcut)

strain_num=0

DO i=0,INT(step_num)

   istrvalue=istr0+step_length*i
   WRITE(filename,'(A7,F6.4,A5)') 'AFFPOS_', istrvalue, '.vasp'

   INQUIRE (FILE=filename, EXIST=filefound)
   IF ( filefound ) THEN

           strain_num=strain_num+1
           strain(strain_num)=istrvalue
           ! To read the POSCAR file
           CALL readpos(filename,        &
                        pos_title,       &
                        alat,            &
                        privect,         &
                        ele_symbol,      &
                        ntype,           &
                        natom,           &
                        natomi,          &
                        ndynamic,        &
                        ncoord,          &
                        pripos,          &
                        prifix,          &
                        atom_type)
           ! To direct
           CALL directpos(alat,     &
                          privect,  &
                          natom,    &
                          ncoord,   &
                          pripos)

           ! To read the file BONDCAR

           OPEN(UNIT=79, FILE='BONDCAR', STATUS='unknown')
           DO ii=1,natom
              READ(79,*) strtmp, bond_num_atom(ii), (dist_atom_indx_tmp(ii,j), atom_atom_indx(ii,j),j=1,bond_num_atom(ii))
              DO j=1,bond_num_atom(ii)
                 READ(dist_atom_indx_tmp(ii,j)(2:4),*) dist_atom_indx(ii,j)
              ENDDO
           ENDDO
           CLOSE(79)
           
           ! To calculation the bond length

           DO ii=1,natom
              DO j=1,bond_num_atom(ii)

                 DO k=1,3
                    dist_direct(ii,j,k)=pripos(ii,k)-pripos(dist_atom_indx(ii,j),k)  &
                                                    +dist_indx(atom_atom_indx(ii,j),k)
                 ENDDO
                 DO k=1,3
                    dist_carte(ii,j,k)=alat*(dist_direct(ii,j,1)*privect(1,k)  &
                                            +dist_direct(ii,j,2)*privect(2,k)  &
                                            +dist_direct(ii,j,3)*privect(3,k))
                 ENDDO

                 str_bond_lgth_atom(ii,j,strain_num)=SQRT(dist_carte(ii,j,1)**2  &
                                                         +dist_carte(ii,j,2)**2  &
                                                         +dist_carte(ii,j,3)**2)
              ENDDO
           ENDDO

   ENDIF
ENDDO

DEALLOCATE(dist_indx)

! To write the file RBOND

OPEN(UNIT=67, FILE='RBOND', STATUS='unknown')
DO i=1,natom
   WRITE(67,'(A5,I03.3,A,I6)') 'ATOM(', i, ')', bond_num_atom(i)
   WRITE(67,*)
   WRITE(67,'(A8,99(A6,I03.3,A))') '  Strain', ('     (',dist_atom_indx(i,j),')',j=1,bond_num_atom(i))

   DO k=1,strain_num
      WRITE(67,'(F8.4,99F10.4)') strain(k), (str_bond_lgth_atom(i,j,k),j=1,bond_num_atom(i))
   ENDDO
   WRITE(67,*)

ENDDO
CLOSE(67)

END SUBROUTINE writebond
