SUBROUTINE writeout
!
!  Purpose:
!
!    To output the result to the file RDAIS, including the calculated energy and stress as a
!    function of strain via affine deformation, or the calculated GSFE as a function of dx
!    and dy, or universal binding energy as a function of dz via alias deformation for 3D and 2D
!    materials
!
!  Record of vevisions:
!      Data         Programmer        Description of change
!      ====         ==========        =====================
!   2017/03/17      Shihao Zhang      Original code
!
!  Variables:
!
!    affcount  : The affcount_th step for affine deformation
!    alicount  : The alicount_th step for alias deformation
!    loopnumx(y,z): The step number for x (y, z) axis
!    Etot0     : TOTEN in OUTCAR file for the initial structure
!
!    Pxx, Pyy, Pzz, Pxy, Pyz, Pzx: Six stress component of structure in OUTCAR file
!    Etot      : TOTEN in OUTCAR file
!    Ece       : Energy without entropy in OUTCAR file
!    Esep      : Energy(sigma->0) in OUTCAR file
!    Ef        : Fermi energy in OUTCAR file
!    Vol       : Volume of structure in OUTCAR file
!    pressure  : PSTRESS in INCAR file
!
!    ten_str   : The weight for tensile strain, i.e., strain(1)=step_length*ten_str
!    shear_str : The weight for shearing strain, i.e., strain(2)=step_length*shear_str
!    istr0     : The initial strain value
!    step_length: The step length for affine deformation
!    step_num  : The step number for affine deformation
!
!    istrvalue : The accumulated strain value
!    stress    : The calculted stress of structure under affine deformation
!
!    istartx(y,z): The initial alias strain value
!    iendx(y,z): The end alias strain value
!    ispacingx(y,z): The spacing value for alias strain
!    zvalue    : The alias deformation will be applied to the atoms, whose z coordinate > zvalue
!    dx(y,z)r  : The displacement along x (y,z) axis
!    GSFE      : The calculated generalized stacking fault energy (GSFE)
!    Ebinding  : The calculated universal binding energy
!    volume    : The volume of POS file
!    area      : The area of a-b plane
!
!
!

USE constants
IMPLICIT NONE

! The variables for reading and writing POS file

INTEGER :: i, j, k, ii, jj, kk
INTEGER :: ntype, natom, ncoord, ndynamic
INTEGER :: natomi(TYPEMAX), atom_type(ATOMMAX), atom_count(TYPEMAX+1)

DOUBLE PRECISION :: alat
DOUBLE PRECISION :: privect(3,3), pripos(ATOMMAX,3)

CHARACTER(len=30) :: inpos_name, outpos_name
CHARACTER(len=30) :: pos_title 
CHARACTER(len=1) :: pricoord, select_dynamic
CHARACTER(len=2) :: ele_symbol(TYPEMAX)
CHARACTER(len=1) :: prifix(ATOMMAX,3)

! The variables for main program

INTEGER :: affcount, alicount
INTEGER :: loopnumx, loopnumy, loopnumz
INTEGER :: ios
DOUBLE PRECISION :: Etot0
DOUBLE PRECISION :: Pxx, Pyy, Pzz, Pxy, Pyz, Pzx
DOUBLE PRECISION :: Vol, Etot, Ece, Esep, Ef, pressure
DOUBLE PRECISION :: ten_str, shear_str, istr0, step_length, step_num
DOUBLE PRECISION :: istrvalue, stress

DOUBLE PRECISION :: istartx, iendx, ispacingx, istarty, iendy, ispacingy, zvalue
DOUBLE PRECISION :: istartz, iendz, ispacingz
DOUBLE PRECISION :: dxr, dyr, dzr
DOUBLE PRECISION :: GSFE, Ebinding
DOUBLE PRECISION :: volume, area, tmp

CHARACTER(len=20) :: titstr1, titstr2
CHARACTER(len=20) :: tmpstr

LOGICAL :: filefound, filefound_rdais

! Whether CONTCAR file exists or Not
! INQUIRE( FILE='CONTCAR', EXIST=filefound )
! IF ( filefound ) THEN
        ! To read the POSCAR file
        inpos_name='CONTCAR'
        CALL readpos(inpos_name,      &
                     pos_title,       &
                     alat,            &
                     privect,         &
                     ele_symbol,      & 
                     ntype,           &
                     natom,           &
                     natomi,          & 
                     ndynamic,        &
                     ncoord,          &
                     pripos,          &
                     prifix,          &
                     atom_type)
! ELSE
! 
!         OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
!         WRITE(19,*)
!         WRITE(19,*) ' ----------------------------------------------------------------------------- '
!         WRITE(19,*) '|                                     ERROR                                   |'
!         WRITE(19,*) '|                            -----------------------                          |'
!         WRITE(19,*) '|    Not find CONTCAR file as writing outfile RDAIS!                          |'
!         WRITE(19,*) ' ----------------------------------------------------------------------------- '
!         CLOSE(19)
!         STOP
! 
! ENDIF

! To read the LOOPCAR
OPEN(UNIT=23, FILE='LOOPCAR', STATUS='unknown')
READ(23,*) titstr1, titstr2
CLOSE(23)

! Whether RDAIS file exists or Not
INQUIRE( FILE='RDAIS', EXIST=filefound )
filefound_rdais=filefound

IF ( filefound ) THEN
        OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown')
        READ(43,*)
        READ(UNIT=43, FMT="(A)", IOSTAT=ios) tmpstr
        IF ( ios .EQ. -1 ) THEN
                filefound_rdais=.FALSE.
        ENDIF
        CLOSE(43)
ENDIF

IF ( .NOT.filefound ) THEN

        IF ( TRIM(titstr1) .EQ. 'AFFINE' ) THEN
                IF ( TRIM(titstr2).EQ.'pure3d' .OR. TRIM(titstr2).EQ.'simp3d' ) THEN
                        ! To write the file RDAIS
                        OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown')
                        WRITE(43,*) ' strain_value        Eng(eV)       Volume(Ang^3)    Stress(GPa)'
                        CLOSE(43)
!                 ELSEIF ( TRIM(titstr2).EQ.'pure2d' .OR. TRIM(titstr2).EQ.'simp2d' ) THEN
!                         ! To write the file RDAIS (2D)
!                         OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown')
!                         WRITE(43,*) ' strain_value        Eng(eV)        Stress(N/m)'
!                         CLOSE(43)
                ELSEIF ( TRIM(titstr2).EQ.'unia2d' .OR. TRIM(titstr2).EQ.'biax2d' ) THEN
                        ! To write the file RDAIS (2D)
                        OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown')
                        WRITE(43,*) ' strain_value        Eng(eV)        Stress(N/m)'
                        CLOSE(43)
                ENDIF
        ELSEIF ( TRIM(titstr1) .EQ. 'ALIAS' ) THEN
                IF ( TRIM(titstr2).EQ.'dxy3d' .OR. TRIM(titstr2).EQ.'dxy2d' ) THEN
                        ! To write the file RDAIS
                        OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown')
                        WRITE(43,*) '      dx          dy            Eng(eV)          GSFE(J/m2)'
                        CLOSE(43)
                ELSEIF ( TRIM(titstr2).EQ.'dz' ) THEN
                        ! To write the file RDAIS
                        OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown')
                        WRITE(43,*) '    dz(Ang)          Eng(eV)        Ebinding(J/m2)'
                        CLOSE(43)                        
                ENDIF
        ENDIF

ENDIF

! Whether CONTCAR file exists or Not
INQUIRE( FILE='OUTCAR', EXIST=filefound )
IF ( filefound ) THEN
        ! TO read OUTCAR
        CALL readoutcar(Pxx, Pyy, Pzz,   &
                        Pxy, Pyz, Pzx,   &
                        Etot, Ece, Esep, &
                        Ef, Vol, pressure)
ELSE

        OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')
        WRITE(19,*)
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        WRITE(19,*) '|                                     ERROR                                   |'
        WRITE(19,*) '|                            -----------------------                          |'
        WRITE(19,*) '|    Not find OUTCAR file as writing outfile RDAIS (ADAIS --output)!          |'
        WRITE(19,*) ' ----------------------------------------------------------------------------- '
        CLOSE(19)
        STOP

ENDIF

! To write the file RDAIS

IF ( TRIM(titstr1) .EQ. 'AFFINE' ) THEN
        ! To read the LOOPCAR
        OPEN(UNIT=23, FILE='LOOPCAR', STATUS='unknown')
        READ(23,*)
        READ(23,*)
        READ(23,*) tmpstr, tmpstr, tmpstr, ten_str
        READ(23,*) tmpstr, tmpstr, tmpstr, shear_str
        READ(23,*) tmpstr, tmpstr, tmpstr, istr0
        READ(23,*) tmpstr, tmpstr, tmpstr, step_length
        READ(23,*) tmpstr, tmpstr, tmpstr, step_num        
        READ(23,*) tmpstr, tmpstr, tmpstr, affcount
        CLOSE(23)
        istrvalue=istr0+step_length*affcount

        IF ( TRIM(titstr2).EQ.'pure3d' .OR. TRIM(titstr2).EQ.'simp3d' ) THEN

                IF ( ABS(shear_str) .LT. ZERO_TOLERANCE ) THEN
                        stress=Pyy*(-1.0000)/10.0000
                ELSE
                        stress=Pxy*(-1.0000)/10.0000
                ENDIF                

                OPEN(UNIT=43, POSITION='Append', FILE='RDAIS')
                WRITE(43,'(F12.6,F20.10,2F16.8)') istrvalue, Etot, Vol, stress
                CLOSE(43)
!         ELSEIF ( TRIM(titstr2).EQ.'pure2d' .OR. TRIM(titstr2).EQ.'simp2d' ) THEN
! 
!                 IF ( ABS(shear_str) .LT. ZERO_TOLERANCE ) THEN
!                         stress=Pyy*(-1.0000)/10.0000
!                 ELSE
!                         stress=Pxy*(-1.0000)/10.0000
!                 ENDIF  
! 
!                 CALL vstruc(volume,  &
!                             alat,    &
!                             privect)
!                 CALL astruc2d(area,  &
!                               alat,  &
!                               privect) 
!                 stress=stress*volume/area*1E-1
!                 OPEN(UNIT=43, POSITION='Append', FILE='RDAIS')
!                 WRITE(43,'(F12.6,F20.10,F16.8)') istrvalue, Etot, stress
!                 CLOSE(43)
        ELSEIF ( TRIM(titstr2).EQ.'unia2d' .OR. TRIM(titstr2).EQ.'biax2d' ) THEN

                IF ( TRIM(titstr2).EQ.'unia2d' ) THEN
                        stress=Pyy*(-1.0000)/10.0000
                ELSEIF ( TRIM(titstr2).EQ.'biax2d' ) THEN
                        stress=(Pxx+Pyy)/2.0000*(-1.0000)/10.0000
                ENDIF

                CALL vstruc(volume,  &
                            alat,    &
                            privect)
                CALL astruc2d(area,  &
                              alat,  &
                              privect) 
                stress=stress*volume/area*1E-1
                OPEN(UNIT=43, POSITION='Append', FILE='RDAIS')
                WRITE(43,'(F12.6,F20.10,F16.8)') istrvalue, Etot, stress
                CLOSE(43)                
        ENDIF
ELSEIF ( TRIM(titstr1) .EQ. 'ALIAS' ) THEN
        IF ( TRIM(titstr2).EQ.'dxy3d' .OR. TRIM(titstr2).EQ.'dxy2d' ) THEN
                ! To read the LOOPCAR
                OPEN(UNIT=23, FILE='LOOPCAR', STATUS='unknown')
                READ(23,*)
                READ(23,*) 
                READ(23,*) tmpstr, tmpstr, tmpstr, istartx
                READ(23,*) tmpstr, tmpstr, tmpstr, iendx
                READ(23,*) tmpstr, tmpstr, tmpstr, ispacingx
                READ(23,*) tmpstr, tmpstr, tmpstr, istarty
                READ(23,*) tmpstr, tmpstr, tmpstr, iendy
                READ(23,*) tmpstr, tmpstr, tmpstr, ispacingy
                READ(23,*) tmpstr, tmpstr, tmpstr, zvalue 
                READ(23,*) tmpstr, tmpstr, tmpstr, alicount
                CLOSE(23)

                ! The number of step_num
                IF ( ABS(iendx-istartx) .LT. ZERO_TOLERANCE .OR.  &
                     (iendx-istartx)*ispacingx .LT. 0.0000  .OR.  &
                     ABS(ispacingx) .LT. ZERO_TOLERANCE ) THEN
                        loopnumx=1
                ELSE
                        loopnumx=FLOOR((iendx-istartx)/ispacingx)+1
                ENDIF
                IF ( ABS(iendy-istarty) .LT. ZERO_TOLERANCE .OR.  &
                     (iendy-istarty)*ispacingy .LT. 0.0000  .OR.  &
                     ABS(ispacingy) .LT. ZERO_TOLERANCE ) THEN
                        loopnumy=1
                ELSE
                        loopnumy=FLOOR((iendy-istarty)/ispacingy)+1
                ENDIF

                dxr=FLOOR((alicount-1.0000)/loopnumy)*ispacingx+istartx
                dyr=MOD(alicount-1,loopnumy)*ispacingy+istarty

                IF ( alicount.EQ.1 .AND. .NOT.filefound_rdais ) THEN
                        GSFE=0.0000
                ELSE
                        OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown') 
                        READ(43,*) 
                        READ(43,*) tmp, tmp, Etot0
                        CLOSE(43)

                        CALL astruc2d(area,  &
                                      alat,  &
                                      privect) 
                        GSFE=(Etot-Etot0)/area*16.0000
                ENDIF
                OPEN(UNIT=43, POSITION='Append', FILE='RDAIS')
                WRITE(43,'(2F12.6,F20.10,F16.8)') dxr, dyr, Etot, GSFE
                CLOSE(43)
        ELSEIF ( TRIM(titstr2).EQ.'dz' ) THEN
                ! To read the LOOPCAR
                OPEN(UNIT=23, FILE='LOOPCAR', STATUS='unknown')
                READ(23,*)
                READ(23,*) 
                READ(23,*) tmpstr, tmpstr, tmpstr, istartz
                READ(23,*) tmpstr, tmpstr, tmpstr, iendz
                READ(23,*) tmpstr, tmpstr, tmpstr, ispacingz
                READ(23,*) tmpstr, tmpstr, tmpstr, zvalue 
                READ(23,*) tmpstr, tmpstr, tmpstr, alicount
                CLOSE(23)

                ! The number of step_num
                IF ( ABS(iendz-istartz) .LT. ZERO_TOLERANCE .OR.  &
                     (iendz-istartz)*ispacingz .LT. 0.0000  .OR.  &
                     ABS(ispacingz) .LT. ZERO_TOLERANCE ) THEN
                        loopnumz=1
                ELSE
                        loopnumz=FLOOR((iendz-istartz)/ispacingz)+1
                ENDIF

                dzr=(alicount-1)*ispacingz+istartz

                IF ( alicount.EQ.1 .AND. .NOT.filefound_rdais ) THEN
                        Ebinding=0.0000
                ELSE
                        OPEN(UNIT=43, FILE='RDAIS', STATUS='unknown') 
                        READ(43,*) 
                        READ(43,*) tmp, Etot0
                        CLOSE(43)

                        CALL astruc2d(area,  &
                                      alat,  &
                                      privect) 
                        Ebinding=(Etot-Etot0)/area*16.0000
                ENDIF
                OPEN(UNIT=43, POSITION='Append', FILE='RDAIS')
                WRITE(43,'(F12.6,F20.10,F16.8)') dzr, Etot, Ebinding
                CLOSE(43)
        ENDIF
ENDIF

! To output the results of Pxx, Pyy, Pzz, Pxy, Pyz, Pzx, Etot, Ece, Esep, Ef, Vol and pressure

OPEN(UNIT=19, POSITION='Append', FILE='OUTFILE')

WRITE(19,*)

IF ( TRIM(titstr1) .EQ. 'AFFINE' ) THEN
        WRITE(19,'(A,I03.3,A1,I03.3)') '  AFFINE DEFORMATION      ', affcount, '/', INT(step_num)
ELSEIF ( TRIM(titstr1) .EQ. 'ALIAS' ) THEN
        IF ( TRIM(titstr2).EQ.'dxy3d' .OR. TRIM(titstr2).EQ.'dxy2d' ) THEN
                WRITE(19,'(A,I04.4,A1,I04.4)') '  ALIAS DEFORMATION      ', alicount, '/', loopnumx*loopnumy
        ELSEIF ( TRIM(titstr2).EQ.'dz' ) THEN
                WRITE(19,'(A,I03.3,A1,I03.3)') '  ALIAS DEFORMATION      ', alicount, '/', loopnumz
        ENDIF
ENDIF

WRITE(19,*) ' ----------------------------------------------------------------------------- '
WRITE(19,'(A,F16.8)') ' | FREE ENERGY TOTEN      : ', Etot
WRITE(19,'(A,F16.8)') ' | ENERGY WITHOUT ENTROPY : ', Ece
WRITE(19,'(A,F16.8)') ' | ENERGY(SIMGA->0)       : ', Esep
WRITE(19,'(A,F16.4)') ' | Pxx (kB)               : ', Pxx
WRITE(19,'(A,F16.4)') ' | Pyy (kB)               : ', Pyy
WRITE(19,'(A,F16.4)') ' | Pzz (kB)               : ', Pzz
WRITE(19,'(A,F16.4)') ' | Pxy (kB)               : ', Pxy
WRITE(19,'(A,F16.4)') ' | Pyz (kB)               : ', Pyz
WRITE(19,'(A,F16.4)') ' | Pzx (kB)               : ', Pzx
WRITE(19,'(A,F16.2)') ' | PSTRESS (kB)           : ', pressure
WRITE(19,'(A,F16.4)') ' | E-FERMI (eV)           : ', Ef
WRITE(19,'(A,F16.4)') ' | VOLUME (Ang^3)         : ', Vol

CLOSE(19)

END SUBROUTINE writeout
