
% ------------------------------------------------------------------------
% Function: pressfield1d(latxz, latyy)
% ------------------------------------------------------------------------
% Aim:
% To solve the hydrostatic stress around dislocation core (GPa) of a dislocation
% ------------------------------------------------------------------------
% Ref: 
% 1/ Ma, Duancheng, et al. Acta Materialia 85 (2015): 53-66.
% ------------------------------------------------------------------------

function [latxz, latyy, pressfld] = pressfield1d (latxz, latyy)

% define global variables

    global BurVect;
    global shear_modulus poisson_ratio;
    global Nmis;
    global Xcoef Xdist Xwid;
    global filepath;
    global mistype;

%   a set of Nmis discrete partial dislocations
% %  the compnents of the two partial dislocation along x axis or z axis
    
    X_coef=Xcoef*BurVect;
    X_dist=Xdist;
    X_wid=Xwid;
    
    if Nmis==1
        X_dist(Nmis)=0;
    else 
        X_dist(Nmis)=0-sum(X_dist(1:(Nmis-1)));
    end
    
% the vector of X and Y that have added the misfit

    if mistype=='e'

        for ii=1:size(latxz,1)
            for jj=1:size(latxz,2)
                if (latyy(ii,jj) > 0.0000)
                    latxz(ii,jj)=latxz(ii,jj)+misfit1d(latxz(ii,jj));
                end
            end
        end
    
    elseif mistype=='s'
        
    end
    
% number of the edge fractional disloctions

    temp=zeros(size(latxz,1),size(latxz,2));

% pressure field produced by the dislocation (unit: GPa)

    for ij=1:size(X_coef,1)*size(X_coef,2)

        temp=temp+X_coef(ij)*(latyy+sign(latyy)*X_wid(ij))./((latxz-X_dist(ij)).^2+(latyy+sign(latyy)*X_wid(ij)).^2);

    end
    
    pressfld=shear_modulus*(1+poisson_ratio)/(3*pi*(1-poisson_ratio))*temp;

%% To plot
    
    scatter(latxz(:),latyy(:),100,pressfld(:));
    
    title('The pressure field in GPa around dislocation core');
    xlabel('Pressfld_Latxz (A)');
    ylabel('Pressfld_Latyy (A)');
    
    set(gca,'LineWidth',2)
    axis equal
    
    % % % Create colorbar
    
    colorbar;
    colormap('jet');

    % % % To save the figure

    figurefile=strcat(filepath,'/pressurefield.fig');
    saveas(1,figurefile);
    close(figure(1));

    fprintf('    >> The figure of pressure field around dislocation core is ploted and saved as "pressurefield.fig" \n');
    
end
