
% ------------------------------------------------------------------------
% Function: pressfield(latxz, latyy)
% ------------------------------------------------------------------------
% Aim:
% To solve the hydrostatic stress around dislocation core (GPa) of a dislocation
% ------------------------------------------------------------------------
% Ref: 
% 1/ Ma, Duancheng, et al. Acta Materialia 85 (2015): 53-66. 
% ------------------------------------------------------------------------
% Note:
% 1/ The input parameters latxx and latyy are in unit of A
% ------------------------------------------------------------------------

function [latxz, latyy, pressfld] = pressfield (latxz, latyy)

% define global variables

    global Dx Wx Dz Wz Ax Az;
    global dx dz;
    global BurVect;
    global Nmis;
    global shear_modulus poisson_ratio;
    global filepath;
    global dis_component_xz;
    global mistype;

% a set of Nmis discrete partial dislocations
% % the compnents of the two partial dislocation along x axis or z axis
    
    if dis_component_xz=='x'

        if Nmis>=1

            % %   Some Matrix

            XZ_coef=zeros(1,Nmis-1);
            XZ_dist=zeros(1,Nmis-1);
            XZ_wid=zeros(1,Nmis);

            for i=1:(Nmis-1)
                XZ_coef(i)=Ax(i);
            end

            if Nmis==1
                XZ_coef(Nmis)=1;
            else 
                XZ_coef(Nmis)=1-sum(XZ_coef(1:(Nmis-1)));
            end

            XZ_coef=XZ_coef*BurVect;

            for i=1:Nmis-1
                XZ_dist(i)=Dx(i);
            end

            if Nmis==1
                XZ_dist(Nmis)=0;
            else 
                XZ_dist(Nmis)=0-sum(XZ_dist(1:(Nmis-1)));
            end

            for i=1:Nmis
                XZ_wid(i)=Wx(i);
            end

        elseif Nmis==0

            XZ_coef=[Ax,(1-Ax)]*BurVect;
            XZ_dist=[-Dx/2,Dx/2];
            XZ_wid=[Wx,Wx];

        elseif Nmis==-1

            XZ_coef=[1/2*(1/2-Ax),Ax,1/2*(1/2-Ax),1/2*(1/2-Ax),Ax,1/2*(1/2-Ax)]*BurVect;
            XZ_dist=[Dx/2+dx,Dx/2,Dx/2-dx,-(Dx/2+dx),-Dx/2,-(Dx/2-dx)];
            XZ_wid=[Wx,Wx,Wx,Wx,Wx,Wx];

        end

    elseif dis_component_xz=='z'
        
        if Nmis>=1

            % %   Some Matrix

            XZ_coef=zeros(1,Nmis-1);
            XZ_dist=zeros(1,Nmis-1);
            XZ_wid=zeros(1,Nmis);

            for i=1:(Nmis-1)
                XZ_coef(i)=Az(i);
            end

            if Nmis==1
                XZ_coef(Nmis)=0;
            else 
                XZ_coef(Nmis)=0-sum(XZ_coef(1:(Nmis-1)));
            end

            XZ_coef=XZ_coef*BurVect;

            for i=1:Nmis-1
                XZ_dist(i)=Dz(i);
            end

            if Nmis==1
                XZ_dist(Nmis)=0;
            else 
                XZ_dist(Nmis)=0-sum(XZ_dist(1:(Nmis-1)));
            end

            for i=1:Nmis
                XZ_wid(i)=Wz(i);
            end            
        
        elseif Nmis==0    
            
            XZ_coef=[Az,-Az]*BurVect;
            XZ_dist=[-Dz/2,Dz/2];
            XZ_wid=[Wz,Wz];
        
        elseif Nmis==-1
            
            XZ_coef=[1/2*(sqrt(3)/6-Az),Az,1/2*(sqrt(3)/6-Az),-1/2*(sqrt(3)/6-Az),-Az,-1/2*(sqrt(3)/6-Az)]*BurVect;
            XZ_dist=[Dz/2+dz,Dz/2,Dz/2-dz,-(Dz/2+dz),-Dz/2,-(Dz/2-dz)];
            XZ_wid=[Wz,Wz,Wz,Wz,Wz,Wz];
            
        end
    
    end
    
% the vector of X and Y that have added the misfit

    if mistype=='e'
        
        for ii=1:size(latxz,1)
            for jj=1:size(latxz,2)
                if (latyy(ii,jj) > 0.0000)
                    latxz(ii,jj)=latxz(ii,jj)+misfit_x(latxz(ii,jj));
                end
            end
        end    
    
    elseif mistype=='s'

        for ii=1:size(latxz,1)
            for jj=1:size(latxz,2)
                if (latyy(ii,jj) > 0.0000)
                    latxz(ii,jj)=latxz(ii,jj)+misfit_z(latxz(ii,jj));
                end
            end
        end 
        
    end
    
% number of the edge fractional disloctions

    temp=zeros(size(latxz,1),size(latxz,2));

% pressure field produced by the dislocation (unit: GPa)

    for ij=1:size(XZ_coef,1)*size(XZ_coef,2)

        temp=temp+XZ_coef(ij)*(latyy+sign(latyy)*XZ_wid(ij))./((latxz-XZ_dist(ij)).^2+(latyy+sign(latyy)*XZ_wid(ij)).^2);

    end
    
    pressfld=shear_modulus*(1+poisson_ratio)/(3*pi*(1-poisson_ratio))*temp;

    % To plot
    
    scatter(latxz(:),latyy(:),100,pressfld(:));
    
    title('The pressure field in GPa around dislocation core');
    xlabel('Pressfld_Latxz (A)');
    ylabel('Pressfld_Latyy (A)');
    
    set(gca,'LineWidth',2)
    axis equal
    
    % % % Create colorbar
    
    colorbar;
    colormap('jet');
    
    % % % To save the figure
    
    figurefile=strcat(filepath,'/pressurefield.fig');
    saveas(1,figurefile);
    close(figure(1));
    
    fprintf('    >> The figure of pressure field around dislocation core is ploted and saved as "pressurefield.fig" \n');
    
end

